;+
; NAME:
;         p3d_display_lineprofiles
;
;         $Id: p3d_display_lineprofiles.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         The purpose of this routine is to provide a graphical user interface
;         in order to view the quality of fits of cross-dispersion line
;         profiles (that are used when performing optimal extraction and doing
;         cross-talk corrections).
;
;         Several input images are used to show the fits. At first the raw data
;         (RAWARRAY; and optionally its variance RAWDARRAY) is drawn for a
;         selected wavelength bin on the cross-dispersion axis (2nd dimension
;         of the images). Thereafter the locations of the spectra are drawn
;         using the trace mask (TRACES; including a spectrum integer
;         identifier). Finally the line profiles (LPROFS) are drawn on top of
;         the other lines.
;
;         The viewer starts up with showing a cross-dispersion cut for the
;         central wavelength bin. The positions of the spectra are indicated
;         with a vertical line and an identifier above every spectrum position.
;         In order to select another wavelength bin you can either enter the
;         bin number (1-based) in the text widget below the draw area to the
;         left. Or you can drag the widget slider to show the bin of interest.
;
;         Error bars of the raw data are shown by pressing the button
;         'Show errors' in the lower left part of the control panel under the
;         draw area. When this button is pressed it changes name to
;         'Hide errors', press it to then remove the error bars. The y-range is
;         set automatically to use the maximum range in the shown x-range.
;         The character size can be changed by selecting another value in the
;         droplist, that says '1.0' when the viewer starts. Press the button
;         'Auto Y-Range -Y-' to always use the maximum across the full x-range.
;         Press the same button to again activate automatic y-ranging.
;
;         The x-range can be shifted by clicking the left and right arrow
;         buttons next to the 'shift' label, or by first clicking in the draw
;         area and then pressing the left and right arrow keys. The shift
;         value can be changed by entering a new value in the text widget
;         between the arrow keys, or by pressing Ctrl-left arrow or Ctrl-right
;         arrow.
;
;         You can zoom into the plot, or out from it, by pressing the up and
;         down arrows right of the 'zoom:' label, respectively. You can also
;         first click in the draw area and then press the up and down arrow
;         keys. In order to change the zoom factor, enter a new value
;         (percentage of current range) in the text widget between the arrow
;         buttons, or press Ctrl-up arrow or Ctrl-down arrow.
;
;         If the special keywords TRCP1 and TRCP2 are set then the tool is
;         setup in a special mode to view how well spectra have been found.
;         For one wavelength bin the data is plotted in the cross-dispersion
;         direction. The location of the initially identified spectra are
;         shown with a '+'-sign. The identified spectra, which were also found
;         to be separated by the right distance are shown with an 'X'-sign.
;         Finally, the location of spectra, which match the pre-defined mask
;         mask REFMASK, are shown with a '*'-sign. For further details, see
;         p3d_tracing_findspec.pro
;
; AUTHOR:
;         Christer Sandin and Joris Gerssen
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: display routine
;
; CALLING SEQUENCE:
;         p3d_display_lineprofiles,rawarray,traces,lprofs,rawdarray=, $
;             title=,filename=,/modal,proffun=,optex=, $
;             topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         rawarray        - A two-dimensional array of decimal type that
;                           contains the raw image used when fitting the cross-
;                           dispersion line profiles. It is expected that the
;                           dispersion axis is in the first dimension of
;                           RAWARRAY (as well as TRACES and RAWDARRAY).
;         traces          - A two-dimensional array of decimal type that
;                           contains the (starting) centroids of every spectrum
;                           at every bin. This array is used to plot the
;                           positions (and number identifier) of every
;                           spectrum.
;         lprofs          - A five-dimensional array of decimal type that
;                           contains the fitted line profile. The array
;                           dimensions 1 and 2 of LPROFS must agree with those
;                           of TRACES. LPROFS is only used if OPTEX is not set.
;
; KEYWORD PARAMETERS:
;         rawdarray       - A two-dimensional array of decimal type that
;                           contains the variance of RAWARRAY. The dimensions
;                           of RAWDARRAY must be the same as those of RAWARRAY.
;         title ['p3d: Cross-dispersion line profiles viewer: "'+FILENAME+'"']
;                         - A scalar string that is used in the viewer title.
;                           Note that FILENAME must be set if TITLE is not set.
;         filename        - A scalar string with the name of the file of
;                           RAWARRAY. FILENAME is only used if TITLE is not
;                           used, and then in order to create a title.
;         modal           - If this keyword is set then the viewer is started
;                           as a modal widget (and thereby blocks IDL until the
;                           widget is closed).
;         proffun         - A scalar string with the name of the function to
;                           use when (re-)calculating the line profile.
;         optex           - A two-dimensional array of decimal type that
;                           contains a secondary profile of every wavelength
;                           bin and spectrum. This variable can be used to
;                           check if the optimal extraction went OK. This
;                           profile is plotted on top of the raw data.
;                           If OPTEX is provided then LPROFS is not used.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         17.07.2009 - A new routine. /CS
;-
PRO p3d_display_lineprofiles_draw,state
  compile_opt hidden,IDL2

  error=0 & rname='p3d_display_lineprofiles_draw: '

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~(*state).debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=(*state).wid.bbase
      catch,/cancel
      return
    endif
  endif ;; ~(*state).debug

  ;;========================================------------------------------
  ;; Draw:

  wset,(*state).wid.idraw

  ;;========================================------------------------------
  ;; Storing plotting variables:

  pm=!p.multi
  !p.multi=0

  ;;========================================------------------------------
  ;; Changing the decomposed state, if necessary:

  device,get_decomposed=decomposed
  if decomposed then device,decomposed=0L

  ;;========================================------------------------------
  ;; Setting up x- and y-ranges:

  data=(*state).rawarray[(*state).wval,*]
  x=lindgen(n_elements(data))+1L
  if (*state).showerrors then ddata=(*state).rawdarray[(*state).wval,*]

  if (*state).autorange then begin
    idx=where(x ge (*state).xrange[0L] and x le (*state).xrange[1L],count)
    yrange=~count?[0.0,1.0]:[0.0,max(data[idx])*1.2]
  endif else begin
    yrange=[0.0,max(data)*1.2]
  endelse

  ;; Optional postscript output:
  for jj=0,keyword_set((*state).debug) do begin 
    if jj then begin
      entrydevice=!d.name & entryfont=!p.font & entrycsize=(*state).charsize
      set_plot,'ps'
      xsize=6.5*2.54 ;; PASP \textwidth
      device,filename='p3d.ps',isolatin=1,/encapsulated,/color, $
             bits_per_pixel=8,xsize=xsize,ysize=7.0
      !p.font=0
      (*state).charsize=0.72
    endif

  cblack=jj?0:1
  cgreen=jj?0:3
  symsize=jj?0.6:1.0

  ;;========================================------------------------------
  ;; Plotting:

  yfac=floor(alog10(yrange[1L])) & syfac=strtrim(yfac,2L)
  yfac=10^double(yfac)
  plot,[0L],/nodata,color=cblack,charsize=(*state).charsize, $
       xrange=(*state).xrange,xtitle='Cross-dispersion axis [px]',xstyle=1, $
       yrange=yrange/yfac,ytitle='Intensity/10!u'+syfac+'!n', $
       ystyle=1,xmargin=[7.0,0.2],ymargin=[3.0,0.2]

  ;; Plotting trace mask positions (with the spectrum number indicated):
  s=size((*state).traces)
  ypos0=yrange[1L]*0.88
  for i=0L,s[2L]-1L do begin
    xpos=(*state).traces[(*state).wval,i]+1.0
    if xpos lt (*state).xrange[0L] or $
       xpos gt (*state).xrange[1L] then continue

    idx=where(x ge xpos-2d0 and x le xpos+2d0,count)
    if count ne 0L then begin
      maxval=max(data[idx])
      oplot,[xpos,xpos],[maxval*1.04,ypos0]/yfac,color=2

      tmpstr=strtrim(i+1L,2L)
      ypos=yrange[1L]*0.9
      xyouts,xpos,ypos/yfac,tmpstr,orientation=90L,color=2, $
             charsize=(*state).charsize
    endif ;; count ne 0L
  endfor ;; i=0L,s[2L]-1L

  if (*state).tmode then begin
    ;; Plotting the indicators of the tracing procedure, which indicate
    ;; successful fits:

    ypos0=yrange[1L]*0.84
    ty=dblarr(n_elements((*state).trcp1))+ypos0
    oplot,(*state).trcp1+1d0,ty/yfac,psym=1L,symsize=symsize,color=cgreen

    ypos0=yrange[1L]*0.86
    ty=dblarr(n_elements((*state).trcp2))+ypos0
    oplot,(*state).trcp2+1d0,ty/yfac,psym=7L,symsize=symsize,color=cgreen

    ypos0=yrange[1L]*0.88
    tx=(*state).traces[(*state).wval,*]+1d0
    ty=dblarr(n_elements(x))+ypos0
    oplot,tx,ty/yfac,psym=1L,symsize=symsize,color=cgreen
    oplot,tx,ty/yfac,psym=7L,symsize=symsize,color=cgreen

  endif ;; (*state).tmode

  ;; Plotting the raw data:
  idx=where(x ge (*state).xrange[0L] and x le (*state).xrange[1L],count)
  if count ne 0L then begin

;   psym=(*state).eprof?1L:10L
    psym=10L
    oplot,x,data/yfac,psym=psym,color=cblack
    if (*state).showerrors then $
      errplot,x,(data-ddata)/yfac,(data+ddata)/yfac,color=cblack,width=0.005

    if (*state).oprof then begin
      ;; Plotting the optimal extraction fits:

      ;; Loop over all rows/spectra:
      halfwidth=((*state).noptex-1L)/2L
      x=dindgen((*state).noptex)-halfwidth

      for i=0L,s[2L]-1L do begin
        center=(*state).lprofs[(*state).wval,i,0L]
        if center lt (*state).xrange[0L] or $
           center gt (*state).xrange[1L] then continue

        arr=(*state).lprofs[(*state).wval,i,*]
        xarr=x+round(center)

        oplot,xarr+1d0,(*state).optex[(*state).wval,i,*]/yfac, $
              color=cgreen,thick=2,psym=10L
      endfor ;; i=0L,s[2L]-1L
    endif else if (*state).eprof then begin
      ;; Plotting the line fits:

      ;; Loop over all rows/spectra and re-create the individual best-fit
      ;; profiles:

      halfwidth=6L & n=2L*halfwidth+1L
      x=dindgen((n-1L)*(*state).proffac+1L)/(*state).proffac
      for i=0L,s[2L]-1L do begin
        center=(*state).lprofs[(*state).wval,i,0L]
        if center lt (*state).xrange[0L] or $
           center gt (*state).xrange[1L] then continue

        arr=(*state).lprofs[(*state).wval,i,*]
        xarr=x+round(center)-halfwidth

        f=p3d_misc_profunction(xarr,arr,proffun=(*state).proffun, $
              logunit=(*state).logunit,topwid=(*state).wid.bbase, $
              error=error,verbose=(*state).verbose)
        if error ne 0 then return

        oplot,xarr+1d0,f/yfac,color=cgreen,thick=2,psym=10L
      endfor ;; i=0L,s[2L]-1L

    endif ;; (*state).eprof
  endif ;; count ne 0L

  ;; Optional postscript output:
    if jj then begin
      device,/close_file
      set_plot,entrydevice & !p.font=entryfont & (*state).charsize=entrycsize
    endif
  endfor

  ;;========================================------------------------------
  ;; Restoring the decomposed state:

  if decomposed then device,decomposed=decomposed

  ;;========================================------------------------------
  ;; Restoring plotting variables:

  !p.multi=pm

  return
END ;;; procedure: p3d_display_lineprofiles_draw


PRO p3d_display_lineprofiles_event,event
  compile_opt hidden,IDL2

  rname='p3d_display_lineprofiles_event: '

  ;;========================================------------------------------
  ;; Getting the state structure:

  widget_control,event.top,get_uvalue=state

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~(*state).debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=event.top
      catch,/cancel
      return
    endif
  endif ;; ~(*state).debug

  ;;========================================------------------------------
  ;; Getting the user value of the event id:

  widget_control,event.id,get_uvalue=uval

  ;; Extracting a widget "index" for those widgets which have defined one:
  index=-1L
  if n_elements(uval) ne 0L  then begin
    tmppos=strpos(uval,'_',/reverse_search)
    if tmppos ne -1L then begin
      index=float(strmid(uval,tmppos+1L))
      uval=strmid(uval,0L,tmppos)
    endif
  endif

  ;;========================================------------------------------
  ;; Handling widget tracking and context events first:

  tmpstrname=strlowcase(tag_names(event,/structure_name))
  if tmpstrname eq 'widget_kbrd_focus' then if event.enter then return

  ;;========================================------------------------------
  ;; Searching for the correct handler for the current event:

  case uval of

    'spectrumplot': begin

      ;; Moving inside the draw widget:
      if tmpstrname eq 'widget_draw' then begin

        if event.press and ~event.modifiers and $
          (event.key eq 5L or event.key eq 6L) then begin

          ;;====================----------
          ;; Left-Right keyboard arrows: shifting up and down
          ;;====================----------

          upordown=event.key eq 5L?0L:1L

          if ~upordown then begin
            ;; left:
            (*state).xrange-=(*state).xshift
          endif else begin
            ;; right:
            (*state).xrange+=(*state).xshift
          endelse

        endif else if event.press and ~event.modifiers and $
          (event.key eq 7L or event.key eq 8L) then begin

          ;;====================----------
          ;; Up-down keyboard arrows: zooming in and out
          ;;====================----------

          inorout=event.key eq 7L?0L:1L
          ztmp=0.5d0*(*state).zoomfac*0.01d0
          factor=inorout?1d0/ztmp:ztmp ;; zooming out, or in

          center=total((*state).xrange)/2d0
          (*state).xrange=center+factor*((*state).xrange-center)

          (*state).xshift*=factor
          widget_control,(*state).wid.tshif,set_value= $
                         strtrim((*state).xshift,2L)

        endif else if event.press and event.modifiers eq 2L and $
          (event.key eq 7L or event.key eq 8L) then begin

          ;;====================----------
          ;; Ctrl+up-down keyboard arrows: changing the zoom factor:
          ;;====================----------

          factor=event.key eq 7L?1d0:-1d0
          tmpq=(*state).zoomfac+factor

          if tmpq gt 0d0 and tmpq lt 1d2 then begin
            (*state).zoomfac=tmpq
            widget_control,(*state).wid.tzoom,set_value=strtrim(tmpq,2L)
          endif

        endif else if event.press and event.modifiers eq 2L and $
          (event.key eq 5L or event.key eq 6L) then begin

          ;;====================----------
          ;; Ctrl+left-right keyboard arrows: changing the shifting factor:
          ;;====================----------

          defvalue=(*state).xshift
          factor=event.key eq 5L?-0.05d0:0.05d0

          tmpq=defvalue*(1d0+factor)

          if round(tmpq) gt 0d0 then begin
            (*state).xshift=tmpq
            widget_control,(*state).wid.tshif,set_value=strtrim(tmpq,2L)
          endif

        endif

        widget_control,(*state).wid.reset,/sensitive
        p3d_display_lineprofiles_draw,state

      endif ;; tmp eq 'widget_draw'
    end ;; case: 'spectrumplot'

    'xreset': begin
      (*state).xrange =(*state).rxrange
      (*state).xshift =(*state).rxshift
      (*state).zoomfac=(*state).rzoomfac

      widget_control,(*state).wid.tshif,set_value=strtrim((*state).xshift,2L)
      widget_control,(*state).wid.tzoom,set_value=strtrim((*state).zoomfac,2L)
      widget_control,(*state).wid.reset,sensitive=0L

      p3d_display_lineprofiles_draw,state

    end ;; case: 'xreset'

    'plot': begin
      widget_control,event.top,/hourglass
      widget_control,event.id,get_value=val
      case index of
        0: tmp=val[0L]
        1: begin
          ;; Text widget:
          defvalue=(*state).wval+1L
          on_ioerror,format_error
          tmp=long(val[0L])
          on_ioerror,NULL
        end
      endcase

      if tmp ge 1L and tmp le (*state).si[1L] and $
         tmp-1L ne (*state).wval then begin

        (*state).wval=tmp-1L

        ;; Updating the related widgets:
        tmp=strtrim((*state).wval+1L,2L)
        widget_control,(*state).wid.tspec,set_value=tmp
        widget_control,(*state).wid.bspsl,set_value=(*state).wval+1L
        sensitive=(*state).wval+1L gt 1L
        widget_control,(*state).wid.bspdc,sensitive=sensitive
        sensitive=(*state).wval+1L lt (*state).si[1L]
        widget_control,(*state).wid.bspic,sensitive=sensitive

        p3d_display_lineprofiles_draw,state

      endif else begin
        ;; Re-setting the value of the widget that created the event:
        if ~index then widget_control,(*state).wid.bspsl, $
          set_value=(*state).wval+1L
        if index eq 1L then widget_control,(*state).wid.tspec, $
          set_value=strtrim((*state).wval+1L,2L)
      endelse

    end ;; case: 'plot'

    'plotbindecrincr': begin

      (*state).wval+=~index?-1L:1L

      ;; Updating the related widgets:
      tmp=strtrim((*state).wval+1L,2L)
      widget_control,(*state).wid.tspec,set_value=tmp
      widget_control,(*state).wid.bspsl,set_value=(*state).wval+1L
      sensitive=(*state).wval+1L gt 1L
      widget_control,(*state).wid.bspdc,sensitive=sensitive
      sensitive=(*state).wval+1L lt (*state).si[1L]
      widget_control,(*state).wid.bspic,sensitive=sensitive

      p3d_display_lineprofiles_draw,state

    end ;; case: 'plotbindecrincr'

    ;;================================================
    ;; Shift the x-range, text widget:

    'xshift': begin
      widget_control,event.id,get_value=val

      defvalue=(*state).xshift
      on_ioerror,format_error
      tmp=double(val[0L])
      on_ioerror,NULL

      range=(*state).xrange[1L]-(*state).xrange[0L]
      if tmp gt 0d0 and tmp lt range and tmp ne defvalue then begin
        (*state).xshift=tmp
        widget_control,(*state).wid.tshif,set_value=strtrim(tmp,2L)
        widget_control,(*state).wid.reset,/sensitive
      endif else $
        widget_control,(*state).wid.tshif,set_value=strtrim(defvalue,2L)
    end ;; case: 'xshift'

    'xsupdown': begin
      upordown=~index?0L:1L
      if ~upordown then begin
        ;; left:
        (*state).xrange-=(*state).xshift
      endif else begin
        ;; right:
        (*state).xrange+=(*state).xshift
      endelse
      widget_control,(*state).wid.reset,/sensitive
      p3d_display_lineprofiles_draw,state
    end ;; case: 'xzoominout'

    ;;================================================
    ;; Change zooming factor, text widget:

    'xzoomfactor': begin
      widget_control,event.id,get_value=val

      defvalue=(*state).zoomfac
      on_ioerror,format_error
      tmp=double(val[0L])
      on_ioerror,NULL

      if tmp gt 0d0 and tmp lt 1d2 and tmp ne defvalue then begin
        (*state).zoomfac=tmp
        widget_control,(*state).wid.tzoom,set_value=strtrim(tmp,2L)
        widget_control,(*state).wid.reset,/sensitive
      endif else $
        widget_control,(*state).wid.tzoom,set_value=strtrim(defvalue,2L)
    end ;; case: 'xzoomfactor'

    'xzoominout': begin
      inorout=index?0L:1L
      ztmp=0.5d0*(*state).zoomfac*0.01d0
      factor=inorout?1d0/ztmp:ztmp ;; zooming out, or in

      (*state).xshift*=factor
      widget_control,(*state).wid.tshif,set_value=strtrim((*state).xshift,2L)

      center=total((*state).xrange)/2d0
      (*state).xrange=center+factor*((*state).xrange-center)

      widget_control,(*state).wid.reset,/sensitive
      p3d_display_lineprofiles_draw,state
    end ;; case: 'xzoominout'

    ;;================================================
    ;; Show/hide error bars button:

    'showerrors': begin
      (*state).showerrors=~(*state).showerrors

      tmp=(*state).showerrors?'Hide errors':'Show errors'
      widget_control,(*state).wid.bserr,set_value=tmp

      p3d_display_lineprofiles_draw,state
    end ;; case: 'showerrors'

    ;;================================================
    ;; Auto y-range:

    'autorange': begin
      (*state).autorange=~(*state).autorange
      tmp=(*state).autorange?'Auto Y-range -Y-':'Auto Y-range -N-'
      widget_control,(*state).wid.bauto,set_value=tmp

      p3d_display_lineprofiles_draw,state
    end ;; case: 'autorange'

    ;;================================================
    ;; Auto y-range:

    'profmult': begin
      widget_control,event.id,get_value=val
      val=val[event.index]
      (*state).proffac=long(val)

      p3d_display_lineprofiles_draw,state
    end ;; case: 'profmult'

    ;;================================================
    ;; Character size:

    'charsize': begin
      widget_control,event.id,get_value=val
      val=val[event.index]
      (*state).charsize=float(val)
      p3d_display_lineprofiles_draw,state
    end ;; case: 'charsize'

    ;;================================================
    ;; Exit:

    'exit': begin
      widget_control,event.top,/destroy
      return
    end ;; case: 'exit'

  endcase ;; uval

  return

format_error:
  widget_control,event.id,set_value=strtrim(defvalue,2L)
  on_ioerror,NULL

  return
END ;;; procedure: p3d_display_lineprofiles_event


PRO p3d_display_lineprofiles_cleanup,id
  compile_opt hidden,IDL2

  ;; Get the state information:
  widget_control,id,get_uvalue=state

  if (*state).verbose ge 3 then $
     print,'p3d_display_lineprofiles: cleanup.'

  if ptr_valid(state) then ptr_free,state

  return
END ;;; end of: p3d_display_lineprofiles_cleanup


PRO p3d_display_lineprofiles,rawarray,traces,lprofs,rawdarray=rawdarray, $
        trcp1=trcp1,trcp2=trcp2,modal=modal,title=title,filename=filename, $
        proffun=proffun_,optex=optex,topwid=topwid,logunit=logunit, $
        verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_display_lineprofiles: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)
  if ~n_elements(topwid) then topwid=0L

  if keyword_set(help) then begin
    doc_library,'p3d_display_lineprofiles'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  si=size(rawarray)
  if ~si[si[0L]+2L] or $
     (si[si[0L]+1L] ge 6L and si[si[0L]+1L] le 11L) then begin
    errmsg='RAWARRAY [1] must be set to a two-dimensional raw image, with ' + $
           'the dispersion axis == 1.'
    goto,error_handler
  endif

  st=size(traces)
  if ~st[st[0L]+2L] or $
     (st[st[0L]+1L] ge 6L and st[st[0L]+1L] le 11L) then begin
    errmsg='TRACES [2] must be set to a two-dimensional raw image, with ' + $
           'the same number of elements on the dispersion axis (1) as th' + $
           'ere are columns in RAWARRAY.'
    goto,error_handler
  endif
  if st[1L] ne si[1L] then begin
    errmsg='The trace mask TRACES does not have the same number of eleme' + $
           'nts in dimension 1 ['+strtrim(st[1L],2L)+'] as RAWARRAY ['+ $
           strtrim(si[2L],2L)+']. Cannot continue!'
    goto,error_handler
  endif

  tmode=~n_elements(trcp1)?0L:1L
  if ~tmode then begin & trcp1=1L & trcp2=1L & endif
  if tmode then begin
    s=size(trcp1)
    if s[0L] ne 1L or (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
      errmsg='TRCP1 must, if set, be a one-dimensional array of decimal type.'
      goto,error_handler
    endif

    s=size(trcp2)
    if s[0L] ne 1L or ~s[s[0L]+2L] or $
      (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
      errmsg='TRCP2 must be set if TRCP1 is; it must be a one-dimensional ' + $
             'array of decimal type.'
      goto,error_handler
    endif
  endif ;; tmode
  

  ecalc=~n_elements(rawdarray)?0L:1L
  if ~n_elements(rawdarray) then rawdarray=1L
  if ecalc then begin
    s=size(rawdarray)
    if ~s[s[0L]+2L] or (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
      errmsg='RAWDARRAY must, if set, be a two-dimensional raw image of th' + $
             'e same size as RAWARRAY.'
      goto,error_handler
    endif
    if s[s[0L]+2L] ne si[si[0L]+2L] or s[1L] ne si[1L] then begin
      errmsg='The variance image RAWDARRAY does not have the same size as ' + $
             'RAWARRAY. Cannot continue!'
      goto,error_handler
    endif
  endif ;; ecalc

  oprof=0L
  s=size(optex)
  if s[s[0L]+2L] ne 0L then begin
    if s[0L] ne 3L or s[1L] ne st[1L] or s[2L] ne st[2L] or $
      (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
      case s[0L] of
        0L: tmp='0.'
        1L: tmp='['+strtrim(s[1L],2L)+'].'
        2L: tmp='['+strtrim(s[1L],2L)+','+strtrim(s[2L],2L)+'].'
        3L: tmp='['+strtrim(s[1L],2L)+','+strtrim(s[2L],2L)+','+ $
                strtrim(s[3L],2L)+'].'
        else: tmp='[error].'
      endcase ;; s[0L]
      errmsg='OPTEX must, if set, be of decimal type and have the followi' + $
             'ng dimensions ['+strtrim(st[1L],2L)+','+strtrim(st[2L],2L)+ $
             ',x] (currently the dimensions are '+tmp
      goto,error_handler
    endif
    noptex=s[3L]
    oprof=1L
  endif ;; s[s[0L]+2L] ne 0L
  if ~n_elements(optex) then optex=1L
  if ~n_elements(noptex) then noptex=0L

  eprof=0L
  s=size(lprofs)
  if s[s[0L]+2L] ne 0L and ~oprof then begin
    sb=size(proffun_)
    if sb[sb[0L]+2L] ne 1L or sb[sb[0L]+1L] ne 7L then begin
      errmsg='PROFFUN must be set to a scalar string w' + $
             'ith the name of the function to use.'
      goto,error_handler
    endif
    proffun=strlowcase(proffun_)

    case proffun of
      'gaussian':      num=5L
      'lorentzian':    num=5L
      'gauss/lorentz': num=6L
      'doublegauss':   num=7L
      else: begin
        errmsg=['PROFFUN must be one of the four options:', $
                '  "gaussian", "lorentzian", "gauss/lorentz", "doublegauss"', $
                ' PROFFUN="'+proffun+'" is not a valid option.']
        goto,error_handler
      end
    endcase ;; proffun

    if s[0L] ne 3L or s[1L] ne st[1L] or s[2L] ne st[2L] or s[3L] ne num or $
      (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
      case s[0L] of
        0L: tmp='0.'
        1L: tmp='['+strtrim(s[1L],2L)+'].'
        2L: tmp='['+strtrim(s[1L],2L)+','+strtrim(s[2L],2L)+'].'
        3L: tmp='['+strtrim(s[1L],2L)+','+strtrim(s[2L],2L)+','+ $
                strtrim(s[3L],2L)+'].'
        else: tmp='[error].'
      endcase ;; s[0L]
      errmsg='LPROFS must, if set, be of decimal type and have the followi' + $
             'ng dimensions ['+strtrim(st[1L],2L)+','+strtrim(st[2L],2L)+ $
             ','+strtrim(num,2L)+'] (currently the dimensions are '+tmp
      goto,error_handler
    endif
    eprof=1L
  endif ;; s[s[0L]+2L] ne 0L
  if ~n_elements(lprofs) then lprofs=1L
  if ~n_elements(proffun) then proffun=''

  if ~n_elements(title) then begin
    s=size(filename)
    if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
      errmsg='FILENAME must be a scalar string.'
      goto,error_handler
    endif
  endif ;; ~n_elements(title)

  if ~n_elements(title) then title='p3d: Cross-dispersion line profiles v' + $
      'iewer: "'+p3d_misc_pathify(filename,/dpath)+'"'
  s=size(title)
  if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
    errmsg='TITLE must be a scalar string.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Setting up various variables:

  xrange=[1L,si[2L]]
  xshift=0.05d0*(xrange[1L]-xrange[0L])


  screensize=get_screen_size()
  xsize=screensize[0L]-50L
  ysize=screensize[1L]/2L ;; Y-size of draw widget
  xoffset=(screensize[0L]-xsize-2*2L)/2L

  ;;========================================------------------------------
  ;; Setting up a widget hierarchy:

  blfile=filepath(subdir=['resource','bitmaps'],'shift_left.bmp')
  brfile=filepath(subdir=['resource','bitmaps'],'shift_right.bmp')
  bufile=filepath(subdir=['resource','bitmaps'],'shift_up.bmp')
  bdfile=filepath(subdir=['resource','bitmaps'],'shift_down.bmp')

  wid={bbase:0L,bexit:0L,ddraw:0L,bcont:0L,bcon2:0L,reset:0L,proffac:0L, $
       tspec:0L,bspdc:0L,bspic:0L,bspsl:0L,idraw:0L,bserr:0L,bchas:0L, $
       bshid:0L,tshif:0L,bshiu:0L,bzooi:0L,tzoom:0L,bzooo:0L,bauto:0L}

  wid.bbase=widget_base(title=title,/column,mbar=mbbarwid, $
      /base_align_center,group_leader=topwid,space=2L,xpad=0L,ypad=2L, $
      xoffset=xoffset,tlb_frame_attr=1L,resource_name='p3d')

  mfilewid=widget_button(mbbarwid,value='File',/menu)
  wid.bexit=widget_button(mfilewid,/separator,value='Exit',uvalue='exit', $
      uname=' ',tracking_events=track,accelerator='Ctrl+Q')

  ;;==============
  ;; Cross-dispersion plot:

  ;; The x-size should be defined by the user:
  wid.ddraw=widget_draw(wid.bbase,xsize=xsize,ysize=ysize, $
      /keyboard_events,retain=2L,uvalue='spectrumplot')

  ;;==============
  ;; Control panel:

  wid.bcont=widget_base(wid.bbase,/row,/base_align_center,space=5L, $
      xpad=2L,ypad=0L)
  if si[1L] gt 1L then begin
    wid.tspec=widget_text(wid.bcont,value=strtrim(0L,2L),/editable, $
        xsize=5L,uvalue='plot_1',resource_name='field',/kbrd_focus_events)

    batmpwid=widget_base(wid.bcont,/row,space=0L,xpad=0L,ypad=0L, $
        /base_align_center)
    wid.bspdc=widget_button(batmpwid,value=blfile,uvalue='plotbindecrincr_0', $
        /bitmap)
    wid.bspsl=widget_slider(batmpwid,/suppress_value,/drag,xsize=xsize-200L, $
        uvalue='plot_0')
    wid.bspic=widget_button(batmpwid,value=brfile,uvalue='plotbindecrincr_1', $
        /bitmap)
  endif ;; si[1L] gt 1L

  wid.bcon2=widget_base(wid.bbase,/row,/base_align_center,space=15L, $
      xpad=2L,ypad=0L)

  wid.bserr=widget_button(wid.bcon2,value='Show errors',uvalue='showerrors', $
      sensitive=ecalc)
  values=string([0.8,0.9,1.0,1.1,1.2,2.0],format='(f3.1)')
  wid.bchas=widget_droplist(wid.bcon2,value=values,uvalue='charsize', $
      title='Character size:')

  wid.bauto=widget_button(wid.bcon2,value='Auto Y-range -Y-', $
      uvalue='autorange')

  batmpwid=widget_base(wid.bcon2,/row,space=0L,xpad=0L,ypad=0L, $
      /base_align_center)
  lpll3wid=widget_label(batmpwid,value='shift:')
  str=strtrim(xshift,2L)
  wid.bshid=widget_button(batmpwid,value=blfile,uvalue='xsupdown_0', $
      /bitmap,resource_name='field')
  wid.tshif=widget_text(batmpwid,value=strtrim(str,2L),/editable, $
      uvalue='xshift',xsize=5,resource_name='field',/kbrd_focus_events)
  wid.bshiu=widget_button(batmpwid,value=brfile,uvalue='xsupdown_1', $
      /bitmap,resource_name='field')

  batmpwid=widget_base(wid.bcon2,/row,space=0L,xpad=0L,ypad=0L, $
      /base_align_center)
  lpll4wid=widget_label(batmpwid,value='zoom:')
  wid.bzooi=widget_button(batmpwid,value=bdfile,uvalue='xzoominout_0', $
      /bitmap,resource_name='field')
  zoomfac=90.0 & tmpstr=string(zoomfac,format='(f5.1)')
  wid.tzoom=widget_text(batmpwid,value=tmpstr,/editable,xsize=5L, $
      uvalue='xzoomfactor',resource_name='field',/kbrd_focus_events)
  wid.bzooo=widget_button(batmpwid,value=bufile,uvalue='xzoominout_1', $
      /bitmap,resource_name='field')

  wid.reset=widget_button(wid.bcon2,value='Reset',uvalue='xreset',sensitive=0L)

  proffac=10L
  if eprof then begin
    values=string([20,10,5,2,1],format='(i2)')
    wid.proffac=widget_droplist(wid.bcon2,value=values,uvalue='profmult', $
        title='Profile display multiplier:')
  endif

  ;;========================================------------------------------
  ;; Realizing the tool, and getting the window index for the draw widget:

  widget_control,wid.bbase,/realize
  widget_control,wid.ddraw,get_value=tmp & wid.idraw=tmp
  wval=si[1L]/2L+(si[1L] mod 2L)-1L
  if si[1L] gt 1L then begin
    widget_control,wid.tspec,set_value=strtrim(wval,2L)
    widget_control,wid.bspsl,set_slider_max=si[1L]
    widget_control,wid.bspsl,set_slider_min=1L
    widget_control,wid.bspsl,set_value=wval+1L
    widget_control,wid.bspdc,sensitive=wval+1L gt 1L
    widget_control,wid.bspic,sensitive=wval+1L lt si[1L]
  endif
  widget_control,wid.bchas,set_droplist_select=2L
  if eprof then widget_control,wid.proffac,set_droplist_select=1L

  ;;========================================------------------------------
  ;; Storing the widget size in a state structure:

  state={rawarray:rawarray,rawdarray:sqrt(rawdarray),traces:traces,si:si, $
         oprof:oprof,optex:optex,noptex:noptex, $
         eprof:eprof,lprofs:lprofs,proffun:proffun,trcp1:trcp1,trcp2:trcp2, $
         wval:wval,xrange:xrange,xshift:xshift,proffac:proffac, $
         zoomfac:zoomfac,rxrange:xrange,rxshift:xshift,rzoomfac:zoomfac, $
         wid:wid,ecalc:ecalc,tmode:tmode,showerrors:0L, $
         logunit:logunit,autorange:1L,charsize:1.0,verbose:verbose,debug:debug}

  ;; Storing state information:
  pstate=ptr_new(state,/no_copy)
  widget_control,wid.bbase,set_uvalue=pstate

  ;;========================================------------------------------
  ;; Drawing the data:

  p3d_display_lineprofiles_draw,pstate

  ;;========================================------------------------------
  ;; Managing the widget tool:

  xmanager,'p3d_display_lineprofiles',wid.bbase,no_block=~keyword_set(modal), $
      cleanup='p3d_display_lineprofiles_cleanup', $
      event_handler='p3d_display_lineprofiles_event'

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_display_lineprofiles
