!/*-----------------------------------------------------------------*/
!/*! 
!  \file f2003arome.f 
!  \brief fortran 2003 binding interface
!         requires a fortran compiler with the standard fortran 2003.
!
!  \author  G. Boue
!           EXOEarths, Centro de Astrofisica, Universidade do Porto.
!
!   Copyright (C) 2012, CAUP
!   email of the author : gwenael.boue@astro.up.pt
!
!   This work has been supported by the European Research Council/European
!   Community under the FP7 through Starting Grant agreement number 239953, as
!   well as from Fundacao para a Ciencia e a Tecnologia (FCT) through program
!   Ciencia 2007 funded by FCT/MCTES (Portugal) and POPH/FSE (EC), and in the
!   form of grants reference PTDC/CTE-AST/098528/2008, SFRH/BPD/71230/2010, and
!   SFRH/BPD/81084/2011.
!
!
!*/
!/*-----------------------------------------------------------------*/

!/*----------------------------------------------------------------*/
!/* License of the file :
!
!    This program is free software: you can redistribute it and/or modify
!    it under the terms of the GNU General Public License as published by
!    the Free Software Foundation, either version 3 of the License, or
!    (at your option) any later version.
!
!    This program is distributed in the hope that it will be useful,
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!*/
!/*----------------------------------------------------------------*/

      module module_arome
      use, intrinsic :: ISO_C_BINDING
    
      implicit none 
    
      interface 
        !/*-------------------------------------------------------------*/
        !/* allocate an orbit structure                                 */
        !/*-------------------------------------------------------------*/
        function arome_new_orbit()                                       &
     &             BIND(C, NAME="arome_new_orbit")
          import C_PTR          
          implicit none
          type(C_PTR)                             :: arome_new_orbit
        end function arome_new_orbit

        !/*-------------------------------------------------------------*/
        !/* set orbital parameters to an orbit                          */
        !/*-------------------------------------------------------------*/
        function arome_set_orbit_eo(per,sma,ecc,om,inc,lambda,orbit)     &
     &             BIND(C, NAME="arome_set_orbit_eo")
          import C_PTR, C_DOUBLE, C_INT
          implicit none
          integer(kind=C_INT)                     :: arome_set_orbit_eo
          real(kind=C_DOUBLE), VALUE, intent(in)  :: per, sma, ecc, om,  &
     &                                               inc, lambda
          type(C_PTR), VALUE, intent(in)          :: orbit
        end function arome_set_orbit_eo

        !/*-------------------------------------------------------------*/
        !/* set orbital parameters to an orbit                          */
        !/*-------------------------------------------------------------*/
        function arome_set_orbit_kh(per,sma,k,h,inc,lambda,orbit)        &
     &             BIND(C, NAME="arome_set_orbit_kh")
          import C_PTR, C_DOUBLE, C_INT
          implicit none
          integer(kind=C_INT)                     :: arome_set_orbit_kh
          real(kind=C_DOUBLE), VALUE, intent(in)  :: per, sma, k, h,     &
     &                                               inc, lambda
          type(C_PTR), VALUE, intent(in)          :: orbit
        end function arome_set_orbit_kh

        !/*-------------------------------------------------------------*/
        !/* get transit time given orbital parameters                   */
        !/*-------------------------------------------------------------*/
        function arome_get_orbit_transit_time(orbit)                     &
     &             BIND(C, NAME="arome_get_orbit_transit_time")
          import C_PTR, C_DOUBLE
          implicit none
          real(kind=C_DOUBLE)            :: arome_get_orbit_transit_time
          type(C_PTR), VALUE, intent(in) :: orbit
        end function arome_get_orbit_transit_time

        !/*-------------------------------------------------------------*/
        !/* get rectangular coordinates of a planet                     */
        !/*-------------------------------------------------------------*/
        function arome_get_orbit_xyz(orbit,t,x,y,z)                      &
     &             BIND(C, NAME="arome_get_orbit_xyz")
          import C_PTR, C_DOUBLE, C_INT
          implicit none
          integer(kind=C_INT)                     :: arome_get_orbit_xyz
          type(C_PTR), VALUE, intent(in)          :: orbit
          real(kind=C_DOUBLE), VALUE, intent(in)  :: t
          real(kind=C_DOUBLE), intent(out)        :: x,y,z
        end function arome_get_orbit_xyz

        !/*-------------------------------------------------------------*/
        !/* get multiple rectangular coordinates of a planet            */
        !/*-------------------------------------------------------------*/
        function arome_mget_orbit_xyz(orbit,t,n,x,y,z)                   &
     &             BIND(C, NAME="arome_mget_orbit_xyz")
          import C_PTR, C_DOUBLE, C_INT
          implicit none
          integer(kind=C_INT)                    :: arome_mget_orbit_xyz
          type(C_PTR), VALUE, intent(in)         :: orbit
          real(kind=C_DOUBLE), dimension(*)      :: t
          integer(kind=C_INT), VALUE, intent(in) :: n
          real(kind=C_DOUBLE), dimension(*)      :: x,y,z
        end function arome_mget_orbit_xyz

        !/*-------------------------------------------------------------*/
        !/* release the memory occupied by the structure orbit          */
        !/*-------------------------------------------------------------*/
        subroutine arome_free_orbit(orbit)                               &
     &               BIND(C, NAME="arome_free_orbit")
          import C_PTR
          implicit none
          type(C_PTR), VALUE, intent(in)  :: orbit
        end subroutine arome_free_orbit



        !/*-------------------------------------------------------------*/
        !/* allocate an arome structure with a quadratic limb-darkening */
        !/*-------------------------------------------------------------*/
        function arome_alloc_quad(u1, u2)                                &
     &             BIND(C, NAME="arome_alloc_quad")
          import C_PTR, C_DOUBLE
          implicit none
          type(C_PTR)                             :: arome_alloc_quad
          real(kind=C_DOUBLE), VALUE, intent(in)  :: u1, u2
        end function arome_alloc_quad
    
        !/*-------------------------------------------------------------*/
        !/* allocate an arome structure with a nonlinear limb-darkening */
        !/*-------------------------------------------------------------*/
        function arome_alloc_nl(c1, c2, c3, c4)                          &
     &             BIND(C, NAME="arome_alloc_nl")
          import C_PTR, C_DOUBLE
          implicit none
          type(C_PTR)                             :: arome_alloc_nl
          real(kind=C_DOUBLE), VALUE, intent(in)  :: c1, c2, c3, c4
        end function arome_alloc_nl
    
        !/*-------------------------------------------------------------*/
        !/* re-initialize the quadratic coefficients
        !   return number of errors */
        !/*-------------------------------------------------------------*/
        function arome_reset_quad(u1, u2, arome)                         &
     &             BIND(C, NAME="arome_reset_quad")
          import C_PTR, C_DOUBLE, C_INT
          implicit none
          integer(kind=C_INT)                     :: arome_reset_quad
          real(kind=C_DOUBLE), VALUE, intent(in)  :: u1, u2
          type(C_PTR), VALUE, intent(in)          :: arome
        end function arome_reset_quad
    
        !/*-------------------------------------------------------------*/
        !/* re-initialize the nonlinear coefficients
        !   return number of errors */
        !/*-------------------------------------------------------------*/
        function arome_reset_nl(c1, c2, c3, c4, arome)                   &
     &             BIND(C, NAME="arome_reset_nl")
          import C_PTR, C_DOUBLE, C_INT
          implicit none
          integer(kind=C_INT)                     :: arome_reset_nl
          real(kind=C_DOUBLE), VALUE, intent(in)  :: c1, c2, c3, c4
          type(C_PTR), VALUE, intent(in)          :: arome
        end function arome_reset_nl
    
        !/*-------------------------------------------------------------*/
        !/* release the memory occupied by the structure */
        !/*-------------------------------------------------------------*/
        subroutine arome_free(arome)                                     &
     &               BIND(C, NAME="arome_free")
          import C_PTR
          implicit none
          type(C_PTR), VALUE, intent(in)  :: arome
        end subroutine arome_free
    
        !/*-------------------------------------------------------------*/
        !/* set the parameters of the line profile
        !   returns the number of errors */
        !/*-------------------------------------------------------------*/
        function arome_set_lineprofile(beta0, Vsini, sigma0, zeta, Kmax, &
     &                 arome) BIND(C, NAME="arome_set_lineprofile")
          import C_PTR, C_DOUBLE, C_INT
          implicit none
          integer(kind=C_INT) :: arome_set_lineprofile
          real(kind=C_DOUBLE), VALUE, intent(in) :: beta0, Vsini,        &
     &                                              sigma0, zeta
          integer(kind=C_INT), VALUE, intent(in) :: Kmax
          type(C_PTR), VALUE, intent(in)         :: arome
        end function arome_set_lineprofile
    
        !/*-------------------------------------------------------------*/
        !/* set the parameters of the planet
        !   returns the number of errors */
        !/*-------------------------------------------------------------*/
        function arome_set_planet(Rp, arome)                             &
     &             BIND(C, NAME="arome_set_planet")
          import C_PTR, C_DOUBLE, C_INT
          implicit none
          integer(kind=C_INT)                    :: arome_set_planet
          real(kind=C_DOUBLE), VALUE, intent(in) :: Rp
          type(C_PTR), VALUE, intent(in)         :: arome
        end function arome_set_planet
    
        !/*-------------------------------------------------------------*/
        !/* initialize constants for the CCF technique
        !   returns the number of errors */
        !/*-------------------------------------------------------------*/
        function arome_init_CCF(arome)                                   &
     &             BIND(C, NAME="arome_init_CCF")
          import C_PTR, C_INT
          implicit none
          integer(kind=C_INT)            :: arome_init_CCF
          type(C_PTR), VALUE, intent(in) :: arome
        end function arome_init_CCF
    
        !/*-------------------------------------------------------------*/
        !/* initialize constants for the iodine cell technique
        !   returns the number of errors */
        !/*-------------------------------------------------------------*/
        function arome_init_iodine(arome)                                &
     &             BIND(C, NAME="arome_init_iodine")
          import C_PTR, C_INT
          implicit none
          integer(kind=C_INT)            :: arome_init_iodine
          type(C_PTR), VALUE, intent(in) :: arome
        end function arome_init_iodine
    
        !/*-------------------------------------------------------------*/
        !/* Computes the flux, the subplanet velocity and dispersion
        !   returns the number of errors */
        !/*-------------------------------------------------------------*/
        function arome_calc_fvpbetap(x, y, z, arome)                     &
     &             BIND(C, NAME="arome_calc_fvpbetap")
          import C_PTR, C_INT, C_DOUBLE
          implicit none
          integer(kind=C_INT)                    :: arome_calc_fvpbetap
          real(kind=C_DOUBLE), VALUE, intent(in) :: x, y, z
          type(C_PTR), VALUE, intent(in)         :: arome
        end function arome_calc_fvpbetap
    
        !/*-------------------------------------------------------------*/
        !/* Computes the RM effect measured by the CCF technique */
        !/*-------------------------------------------------------------*/
        function arome_get_RM_CCF(arome)                                 &
     &             BIND(C, NAME="arome_get_RM_CCF")
          import C_PTR, C_DOUBLE
          implicit none
          real(kind=C_DOUBLE)            :: arome_get_RM_CCF
          type(C_PTR), VALUE, intent(in) :: arome
        end function arome_get_RM_CCF
    
        !/*-------------------------------------------------------------*/
        !/* Computes the RM effect measured by the iodine cell technique */
        !/*-------------------------------------------------------------*/
        function arome_get_RM_iodine(arome)                              &
     &             BIND(C, NAME="arome_get_RM_iodine")
          import C_PTR, C_DOUBLE
          implicit none
          real(kind=C_DOUBLE)            :: arome_get_RM_iodine
          type(C_PTR), VALUE, intent(in) :: arome
        end function arome_get_RM_iodine
    
        !/*-------------------------------------------------------------*/
        !/* Computes the RM effect measured the weighted mean */
        !/*-------------------------------------------------------------*/
        function arome_get_RM_mean(arome)                                &
     &             BIND(C, NAME="arome_get_RM_mean")
          import C_PTR, C_DOUBLE
          implicit none
          real(kind=C_DOUBLE)            :: arome_get_RM_mean
          type(C_PTR), VALUE, intent(in) :: arome
        end function arome_get_RM_mean
    
        !/*-------------------------------------------------------------*/
        !/* Computes the RM effect measured by the CCF technique
        !   returns the number of errors */
        !/*-------------------------------------------------------------*/
        function arome_get_RM_CCF_e(arome, res)                          &
     &             BIND(C, NAME="arome_get_RM_CCF_e")
          import C_PTR, C_DOUBLE, C_INT
          implicit none
          integer(kind=C_INT)              :: arome_get_RM_CCF_e
          type(C_PTR), VALUE, intent(in)   :: arome
          real(kind=C_DOUBLE), intent(out) :: res
        end function arome_get_RM_CCF_e
    
        !/*-------------------------------------------------------------*/
        !/* Computes the RM effect measured by the iodine cell technique
        !   returns the number of errors */
        !/*-------------------------------------------------------------*/
        function arome_get_RM_iodine_e(arome, res)                       &
     &             BIND(C, NAME="arome_get_RM_iodine_e")
          import C_PTR, C_DOUBLE, C_INT
          implicit none
          integer(kind=C_INT)              :: arome_get_RM_iodine_e
          type(C_PTR), VALUE, intent(in)   :: arome
          real(kind=C_DOUBLE), intent(out) :: res
        end function arome_get_RM_iodine_e
    
        !/*-------------------------------------------------------------*/
        !/* Computes the RM effect measured by the weighted mean
        !   returns the number of errors */
        !/*-------------------------------------------------------------*/
        function arome_get_RM_mean_e(arome, res)                         &
     &             BIND(C, NAME="arome_get_RM_mean_e")
          import C_PTR, C_DOUBLE, C_INT
          implicit none
          integer(kind=C_INT)              :: arome_get_RM_mean_e
          type(C_PTR), VALUE, intent(in)   :: arome
          real(kind=C_DOUBLE), intent(out) :: res
        end function arome_get_RM_mean_e
    
        !/*-------------------------------------------------------------*/
        !/* Returns the flux previously computed */
        !/*-------------------------------------------------------------*/
        function arome_get_flux(arome)                                   &
     &             BIND(C, NAME="arome_get_flux")
          import C_PTR, C_DOUBLE
          implicit none
          real(kind=C_DOUBLE)            :: arome_get_flux
          type(C_PTR), VALUE, intent(in) :: arome
        end function arome_get_flux
    
        !/*-------------------------------------------------------------*/
        !/* Returns the subplanet velocity previously computed */
        !/*-------------------------------------------------------------*/
        function arome_get_vp(arome)                                     &
     &             BIND(C, NAME="arome_get_vp")
          import C_PTR, C_DOUBLE
          implicit none
          real(kind=C_DOUBLE)            :: arome_get_vp
          type(C_PTR), VALUE, intent(in) :: arome
        end function arome_get_vp
    
        !/*-------------------------------------------------------------*/
        !/* Returns the subplanet radial dispersion previously computed */
        !/*-------------------------------------------------------------*/
        function arome_get_betapR(arome)                                 &
     &             BIND(C, NAME="arome_get_betapR")
          import C_PTR, C_DOUBLE
          implicit none
          real(kind=C_DOUBLE)            :: arome_get_betapR
          type(C_PTR), VALUE, intent(in) :: arome
        end function arome_get_betapR
    
        !/*-------------------------------------------------------------*/
        !/* Returns the subplanet tangential dispersion previously computed */
        !/*-------------------------------------------------------------*/
        function arome_get_betapT(arome)                                 &
     &             BIND(C, NAME="arome_get_betapT")
          import C_PTR, C_DOUBLE
          implicit none
          real(kind=C_DOUBLE)            :: arome_get_betapT
          type(C_PTR), VALUE, intent(in) :: arome
        end function arome_get_betapT
    
        !/*-------------------------------------------------------------*/
        !/* Set manually the flux */
        !/*-------------------------------------------------------------*/
        function arome_set_flux(flux, arome)                             &
     &             BIND(C, NAME="arome_set_flux")
          import C_PTR, C_DOUBLE, C_INT
          implicit none
          integer(kind=C_INT)                    :: arome_set_flux
          real(kind=C_DOUBLE), VALUE, intent(in) :: flux
          type(C_PTR), VALUE, intent(in)         :: arome
        end function arome_set_flux
    
        !/*-------------------------------------------------------------*/
        !/* Set manually the subplanet velocity */
        !/*-------------------------------------------------------------*/
        function arome_set_vp(vp, arome)                                 &
     &             BIND(C, NAME="arome_set_vp")
          import C_PTR, C_DOUBLE, C_INT
          implicit none
          integer(kind=C_INT)                    :: arome_set_vp
          real(kind=C_DOUBLE), VALUE, intent(in) :: vp
          type(C_PTR), VALUE, intent(in)         :: arome
        end function arome_set_vp
    
        !/*-------------------------------------------------------------*/
        !/* Set manually the subplanet radial dispersion */
        !/*-------------------------------------------------------------*/
        function arome_set_betapR(betapR, arome)                         &
     &             BIND(C, NAME="arome_set_betapR")
          import C_PTR, C_DOUBLE, C_INT
          implicit none
          integer(kind=C_INT)                    :: arome_set_betapR
          real(kind=C_DOUBLE), VALUE, intent(in) :: betapR
          type(C_PTR), VALUE, intent(in)         :: arome
        end function arome_set_betapR
    
        !/*-------------------------------------------------------------*/
        !/* Set manually the subplanet tangential dispersion */
        !/*-------------------------------------------------------------*/
        function arome_set_betapT(betapT, arome)                         &
     &             BIND(C, NAME="arome_set_betapT")
          import C_PTR, C_DOUBLE, C_INT
          implicit none
          integer(kind=C_INT)                    :: arome_set_betapT
          real(kind=C_DOUBLE), VALUE, intent(in) :: betapT
          type(C_PTR), VALUE, intent(in)         :: arome
        end function arome_set_betapT
    
        !/*-------------------------------------------------------------*/
        !/* Force the program to exit after an error */
        !/*-------------------------------------------------------------*/
        subroutine arome_set_exit_on_error()                             &
     &               BIND(C, NAME="arome_set_exit_on_error")
          implicit none
        end subroutine arome_set_exit_on_error
    
        !/*-------------------------------------------------------------*/
        !/* Force the program to continue after an error */
        !/*-------------------------------------------------------------*/
        subroutine arome_set_continue_on_error()                         &
     &               BIND(C, NAME="arome_set_continue_on_error")
          implicit none
        end subroutine arome_set_continue_on_error

        !/*-------------------------------------------------------------*/
        !/* run User function on error */
        !/*-------------------------------------------------------------*/
        subroutine arome_set_func_on_error(userfunch)                    &
     &               BIND(C, NAME="arome_set_func_on_error")
          import C_FUNPTR
          implicit none
          type(C_FUNPTR), VALUE, intent(in) :: userfunch
        end subroutine arome_set_func_on_error
        
    
        !/*-------------------------------------------------------------*/
        !/* Set the internal parameter of the Numerical Recipes gcf function */
        !/*-------------------------------------------------------------*/
        function arome_set_gcf_ITMAX(ITMAX, arome)                       &
     &             BIND(C, NAME="arome_set_gcf_ITMAX")
          import C_PTR, C_INT
          implicit none
          integer(kind=C_INT)                    :: arome_set_gcf_ITMAX
          integer(kind=C_INT), VALUE, intent(in) :: ITMAX
          type(C_PTR), VALUE, intent(in)         :: arome
        end function arome_set_gcf_ITMAX
    
        !/*-------------------------------------------------------------*/
        !/* Set the internal parameter of the Numerical Recipes gcf function */
        !/*-------------------------------------------------------------*/
        function arome_set_gcf_EPS(EPS, arome)                           &
     &             BIND(C, NAME="arome_set_gcf_EPS")
          import C_PTR, C_INT, C_DOUBLE
          implicit none
          integer(kind=C_INT)                    :: arome_set_gcf_EPS
          real(kind=C_DOUBLE), VALUE, intent(in) :: EPS
          type(C_PTR), VALUE, intent(in)         :: arome
        end function arome_set_gcf_EPS
    
        !/*-------------------------------------------------------------*/
        !/* Set the internal parameter of the Numerical Recipes gcf function */
        !/*-------------------------------------------------------------*/
        function arome_set_gcf_FPMIN(FPMIN, arome)                       &
     &             BIND(C, NAME="arome_set_gcf_FPMIN")
          import C_PTR, C_INT, C_DOUBLE
          implicit none
          integer(kind=C_INT)                    :: arome_set_gcf_FPMIN
          real(kind=C_DOUBLE), VALUE, intent(in) :: FPMIN
          type(C_PTR), VALUE, intent(in)         :: arome
        end function arome_set_gcf_FPMIN
    
        !/*-------------------------------------------------------------*/
        !/* Set the internal parameter of the Numerical Recipes gser function */
        !/*-------------------------------------------------------------*/
        function arome_set_gser_ITMAX(ITMAX, arome)                      &
     &             BIND(C, NAME="arome_set_gser_ITMAX")
          import C_PTR, C_INT
          implicit none
          integer(kind=C_INT)                    :: arome_set_gser_ITMAX
          integer(kind=C_INT), VALUE, intent(in) :: ITMAX
          type(C_PTR), VALUE, intent(in)         :: arome
        end function arome_set_gser_ITMAX
    
        !/*-------------------------------------------------------------*/
        !/* Set the internal parameter of the Numerical Recipes gser function */
        !/*-------------------------------------------------------------*/
        function arome_set_gser_EPS(EPS, arome)                          &
     &             BIND(C, NAME="arome_set_gser_EPS")
          import C_PTR, C_INT, C_DOUBLE
          implicit none
          integer(kind=C_INT)                    :: arome_set_gser_EPS
          real(kind=C_DOUBLE), VALUE, intent(in) :: EPS
          type(C_PTR), VALUE, intent(in)         :: arome
        end function arome_set_gser_EPS
    
        !/*-------------------------------------------------------------*/
        !/* Set the internal parameter of the Numerical Recipes qtrap function */
        !/*-------------------------------------------------------------*/
        function arome_set_qtrap_EPS(EPS, arome)                         &
     &             BIND(C, NAME="arome_set_qtrap_EPS")
          import C_PTR, C_INT, C_DOUBLE
          implicit none
          integer(kind=C_INT)                    :: arome_set_qtrap_EPS
          real(kind=C_DOUBLE), VALUE, intent(in) :: EPS
          type(C_PTR), VALUE, intent(in)         :: arome
        end function arome_set_qtrap_EPS
    
        !/*-------------------------------------------------------------*/
        !/* Set the internal parameter of the Numerical Recipes qtrap function */
        !/*-------------------------------------------------------------*/
        function arome_set_qtrap_JMAX(JMAX, arome)                       &
     &             BIND(C, NAME="arome_set_qtrap_JMAX")
          import C_PTR, C_INT
          implicit none
          integer(kind=C_INT)                    :: arome_set_qtrap_JMAX
          integer(kind=C_INT), VALUE, intent(in) :: JMAX
          type(C_PTR), VALUE, intent(in)         :: arome
        end function arome_set_qtrap_JMAX
    
    
        !/*-------------------------------------------------------------*/
        !/* Allocate memory for vectorial call to arome routines */
        !/*-------------------------------------------------------------*/
        function arome_malloc(n, arome)                                  &
     &             BIND(C, NAME="arome_malloc")
          import C_PTR, C_INT
          implicit none
          integer(kind=C_INT)                    :: arome_malloc
          integer(kind=C_INT), VALUE, intent(in) :: n
          type(C_PTR), VALUE, intent(in)         :: arome
        end function arome_malloc
    
        !/*-------------------------------------------------------------*/
        !/* Free memory for vectorial call to arome routines */
        !/*-------------------------------------------------------------*/
        subroutine arome_mfree(arome)                                    &
     &               BIND(C, NAME="arome_mfree")
          import C_PTR
          implicit none
          type(C_PTR), VALUE, intent(in) :: arome
        end subroutine
    
        !/*-------------------------------------------------------------*/
        !/* Multiple call to arome_fvpbetap */
        !/*-------------------------------------------------------------*/
        function arome_mcalc_fvpbetap(x, y, z, n, arome)                 &
     &             BIND(C, NAME="arome_mcalc_fvpbetap")
          import C_PTR, C_DOUBLE, C_INT
          implicit none
          integer(kind=C_INT) :: arome_mcalc_fvpbetap
          real(kind=C_DOUBLE), dimension(*)      :: x, y, z
          integer(kind=C_INT), VALUE, intent(in) :: n
          type(C_PTR), VALUE, intent(in) :: arome
        end function arome_mcalc_fvpbetap
    
        !/*-------------------------------------------------------------*/
        !/* Multiple call to arome_get_flux */
        !/*-------------------------------------------------------------*/
        function arome_mget_flux(arome, n, flux)                         &
     &             BIND(C, NAME="arome_mget_flux")
          import C_PTR, C_INT, C_DOUBLE
          implicit none
          integer(kind=C_INT)                    :: arome_mget_flux
          type(C_PTR), VALUE, intent(in)         :: arome
          integer(kind=C_INT), VALUE, intent(in) :: n
          !type(C_PTR), VALUE                     :: flux
          real(kind=C_DOUBLE), dimension(*)      :: flux
        end function arome_mget_flux
    
        !/*-------------------------------------------------------------*/
        !/* Multiple call to arome_get_vp */
        !/*-------------------------------------------------------------*/
        function arome_mget_vp(arome, n, vp)                             &
     &             BIND(C, NAME="arome_mget_vp")
          import C_PTR, C_INT, C_DOUBLE
          implicit none
          integer(kind=C_INT)                    :: arome_mget_vp
          type(C_PTR), VALUE, intent(in)         :: arome
          integer(kind=C_INT), VALUE, intent(in) :: n
          !type(C_PTR), VALUE                     :: vp
          real(kind=C_DOUBLE), dimension(*)      :: vp
        end function arome_mget_vp
    
        !/*-------------------------------------------------------------*/
        !/* Multiple call to arome_get_betapR */
        !/*-------------------------------------------------------------*/
        function arome_mget_betapR(arome, n, betapR)                     &
     &             BIND(C, NAME="arome_mget_betapR")
          import C_PTR, C_INT, C_DOUBLE
          implicit none
          integer(kind=C_INT)                    :: arome_mget_betapR
          type(C_PTR), VALUE, intent(in)         :: arome
          integer(kind=C_INT), VALUE, intent(in) :: n
          !type(C_PTR), VALUE                     :: betapR
          real(kind=C_DOUBLE), dimension(*)      :: betapR
        end function arome_mget_betapR
    
        !/*-------------------------------------------------------------*/
        !/* Multiple call to arome_get_betapT */
        !/*-------------------------------------------------------------*/
        function arome_mget_betapT(arome, n, betapT)                     &
     &             BIND(C, NAME="arome_mget_betapT")
          import C_PTR, C_INT, C_DOUBLE
          implicit none
          integer(kind=C_INT)                    :: arome_mget_betapT
          type(C_PTR), VALUE, intent(in)         :: arome
          integer(kind=C_INT), VALUE, intent(in) :: n
          !type(C_PTR), VALUE                     :: betapT
          real(kind=C_DOUBLE), dimension(*)      :: betapT
        end function arome_mget_betapT
    
        !/*-------------------------------------------------------------*/
        !/* Multiple call to arome_set_flux */
        !/*-------------------------------------------------------------*/
        function arome_mset_flux(flux, n, arome)                         &
     &             BIND(C, NAME="arome_mset_flux")
          import C_PTR, C_INT, C_DOUBLE
          implicit none
          integer(kind=C_INT)                    :: arome_mset_flux
          real(kind=C_DOUBLE), dimension(*)      :: flux
          integer(kind=C_INT), VALUE, intent(in) :: n
          type(C_PTR), VALUE, intent(in)         :: arome
        end function arome_mset_flux
    
        !/*-------------------------------------------------------------*/
        !/* Multiple call to arome_set_vp */
        !/*-------------------------------------------------------------*/
        function arome_mset_vp(vp, n, arome)                             &
     &             BIND(C, NAME="arome_mset_vp")
          import C_PTR, C_INT, C_DOUBLE
          implicit none
          integer(kind=C_INT)                    :: arome_mset_vp
          real(kind=C_DOUBLE), dimension(*)      :: vp
          integer(kind=C_INT), VALUE, intent(in) :: n
          type(C_PTR), VALUE, intent(in)         :: arome
        end function arome_mset_vp
    
        !/*-------------------------------------------------------------*/
        !/* Multiple call to arome_set_betapR */
        !/*-------------------------------------------------------------*/
        function arome_mset_betapR(betapR, n, arome)                     &
     &             BIND(C, NAME="arome_mset_betapR")
          import C_PTR, C_INT, C_DOUBLE
          implicit none
          integer(kind=C_INT)                    :: arome_mset_betapR
          real(kind=C_DOUBLE), dimension(*)      :: betapR
          integer(kind=C_INT), VALUE, intent(in) :: n
          type(C_PTR), VALUE, intent(in)         :: arome
        end function arome_mset_betapR
    
        !/*-------------------------------------------------------------*/
        !/* Multiple call to arome_set_betapT */
        !/*-------------------------------------------------------------*/
        function arome_mset_betapT(betapT, n, arome)                     &
     &             BIND(C, NAME="arome_mset_betapT")
          import C_PTR, C_INT, C_DOUBLE
          implicit none
          integer(kind=C_INT)                    :: arome_mset_betapT
          real(kind=C_DOUBLE), dimension(*)      :: betapT
          integer(kind=C_INT), VALUE, intent(in) :: n
          type(C_PTR), VALUE, intent(in)         :: arome
        end function arome_mset_betapT
        
        !/*-------------------------------------------------------------*/
        !/* Multiple call to arome_get_RM_CCF_e */
        !/*-------------------------------------------------------------*/
        function arome_mget_RM_CCF(arome, n, res)                        &
     &             BIND(C, NAME="arome_mget_RM_CCF")
          import C_PTR, C_INT, C_DOUBLE
          implicit none
          integer(kind=C_INT)                    :: arome_mget_RM_CCF
          type(C_PTR), VALUE, intent(in)         :: arome
          integer(kind=C_INT), VALUE, intent(in) :: n
          !type(C_PTR), VALUE                     :: res
          real(kind=C_DOUBLE), dimension(*)      :: res
        end function arome_mget_RM_CCF
        
        !/*-------------------------------------------------------------*/
        !/* Multiple call to arome_get_RM_iodine_e */
        !/*-------------------------------------------------------------*/
        function arome_mget_RM_iodine(arome, n, res)                     &
     &             BIND(C, NAME="arome_mget_RM_iodine")
          import C_PTR, C_INT, C_DOUBLE
          implicit none
          integer(kind=C_INT)                    :: arome_mget_RM_iodine
          type(C_PTR), VALUE, intent(in)         :: arome
          integer(kind=C_INT), VALUE, intent(in) :: n
          !type(C_PTR), VALUE                     :: res
          real(kind=C_DOUBLE), dimension(*)      :: res
        end function arome_mget_RM_iodine
        
        !/*-------------------------------------------------------------*/
        !/* Multiple call to arome_get_RM_mean_e */
        !/*-------------------------------------------------------------*/
        function arome_mget_RM_mean(arome, n, res)                       &
     &             BIND(C, NAME="arome_mget_RM_mean")
          import C_PTR, C_INT, C_DOUBLE
          implicit none
          integer(kind=C_INT)                    :: arome_mget_RM_mean
          type(C_PTR), VALUE, intent(in)         :: arome
          integer(kind=C_INT), VALUE, intent(in) :: n
          !type(C_PTR), VALUE                     :: res
          real(kind=C_DOUBLE), dimension(*)      :: res
        end function arome_mget_RM_mean
    
      end interface
      end module module_arome
