;+
; NAME:
;         p3d_misc_dataclip
;
;         $Id: p3d_misc_dataclip.pro 83 2010-03-05 11:33:06Z olestreicher $
;
; PURPOSE:
;         This routine takes an input array and calculates a limited range of
;         values using a histogram function and a percentage. For instance, if
;         the percentage is 95.0 (95%) then the lowermost 2.5% and uppermost
;         2.5% of all pixel values are left outside the limit (RANGE).
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_dataclip,array,range,percentage=, $
;             topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         array           - An array of decimal type. This is the data array
;                           that is used to calculate the range values. It can
;                           have any dimensions.
;
; KEYWORD PARAMETERS:
;         percentage [95.0] - 
;                           RANGE is calculated to embrace with many percent
;                           of all pixel values in ARRAY.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         range           - A two-element array returning the lower and upper
;                           limits calculated in this routine.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         19.06.2009 - Corrected for a bug in the IDL function HISTOGRAM. /CS
;         16.11.2008 - Initial programming. /CS
;-
PRO p3d_misc_dataclip,array,range,percentage=percentage_,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL version <6.2, cannot continue!.'
  error=0 & rname='p3d_misc_dataclip: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_misc_dataclip'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  s=size(array)
  if ~s[0L] or (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='ARRAY must be set to an array of floating point type.'
    goto,error_handler
  endif

  if ~n_elements(n) then n=1000L

  percentage=~n_elements(percentage_)?95.0:percentage_

  on_ioerror,second_try
  notok=1L
  percentage=float(percentage)

  notok=0L
second_try:
  if notok then begin
    on_ioerror,givingup
    percentage=strmid(percentage,0L,strlen(percentage)-1L)
    percentage=float(percentage)
    notok=0L
  endif

givingup: 
  on_ioerror,NULL
  if notok then begin
    errmsg='PERCENTAGE must be a decimal scalar or a string of type "x.xx%".'
    goto,error_handler
  endif

  if percentage le 0.0 or percentage gt 100. then begin
    errmsg='PERCENTAGE must be a decimal scalar or a string of type "x.x' + $
           'x%".; 0<percentage<=100.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Calculating the clipping values:

  range=[0.0,0.0]

  idx=where(array gt 0.0,count)
  if count ne 0L then begin

    if percentage eq 100.0 then begin
      mini=min(array[idx],max=tmp)
      range=[mini,tmp]
    endif else begin

      data=array[idx]
      mini=min(data,max=maxi)
      binsize=(maxi-mini)/float(n-1L)
      xarr=findgen(n+1L)*binsize+mini & revxarr=reverse(xarr)

      ;; Calculating the minimum and maximum values:
      ;; NOTE! If MAX is specified instead of BINSIZE histogram does not
      ;;       return the number of elements it claims it will...
      ;;       (this is because of a bug in IDL!)
      hist=histogram(data,binsize=binsize,nbins=n,min=mini);,max=maxi)
      hist[0L]=0L
      revhist=reverse(hist)

      thist=total(hist,/cumulative,/double) & maxhist=thist[n-1L]/double(n)
      rhist=total(revhist,/cumulative,/double)

      perm=n/1d2*double(percentage)
      x=0.5d0*(double(n)-perm)

      if maxhist*x lt 1d0 then begin
        range=[mini,maxi]
        return
      endif

      ;; Calculating the lower limit:
      sum=0d0 & value=maxhist*x & i=-1L
      while sum lt value do begin
        sum+=hist[++i]
      endwhile
      i0=i-1 & i1=i
      minval=xarr[i0]+(xarr[i0]-xarr[i1])*(value-thist[i0])/ $
             (thist[i0]-thist[i1])

      ;; Calculating the upper limit:
      sum=0d0 & value=maxhist*x & i=-1L
      while sum lt value do begin
        sum+=revhist[++i]
      endwhile
      i0=i-1L & i1=i
      maxval=revxarr[i0]+(revxarr[i0]-revxarr[i1])*(value    -rhist[i0])/ $
             (rhist[i0]-rhist[i1])

      range=[minval,maxval]

    endelse ;; percentage eq 100.0
  endif ;; count ne 0L

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_misc_dataclip
