;+
; NAME:
;         p3d_misc_read_params
;
;         $Id: p3d_misc_read_params.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         For an input string array (PARNAME), with a corresponding array of
;         values (PARVALUE), the string NAME is searched for, and the
;         corresponding value is returned in VALUE.
;
;         The read parameter is, optionally, converted to an integer or a
;         floating point value.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_read_params,parname,parvalue,name,value,must_exist=, $
;             uparname=,uparvalue=,found=,type=,/upo,/nou,/a0,topwid=, $
;             logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         parname         - List of parameter names. Both upper and lower case
;                           are ok.
;         parvalue        - List of parameter values, corresponding to
;                           PARNAME.
;         name            - The parameter name for which the value is
;                           requested. Both upper and lower case are ok.
;
; KEYWORD PARAMETERS:
;         must_exist      - If this keyword is set and a parameter is not,
;                           found, then an error is issued.
;         uparname        - A list of alternative parameter names; both upper
;                           and lower case entries are ok. If UPARNAME and
;                           UPARVALUE exist, then the parameter is replaced
;                           with the value of UPARVALUE, if found.
;         uparvalue       - A list of alternative parameter values,
;                           corresponding to PARNAME; cf. UPARNAME.
;         found           - This keyword is set to 1 if the parameter was
;                           found, otherwise it is set to 0.
;         type            - If this scalar string is set then the read
;                           parameter is checked if it conforms to the variable
;                           type. TYPE can be set to 'integer' or 'float'. If
;                           TYPE is not set then a string is returned.
;         upo             - If this keyword is set then the parameter origin
;                           is set to '[userparfile]'.
;         nou             - If this keyword is set then no parameter origin is
;                           set (useful with parameters which are not allowed
;                           to change value).
;         a0              - If set, then routine returns if PARNAME is
;                           undefined.
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         value           - Contains the value of the parameter with the
;                           name NAME. If NAME is not found then -1L is
;                           returned.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         06.10.2008 - Converted from the original routine
;                      read_p3d_online_params of Thomas Becker. /CS
;-
PRO p3d_misc_read_params,parname,parvalue,name,value,must_exist=must_exist, $
        uparname=uparname,uparvalue=uparvalue,found=found,type=type_,upo=upo, $
        nou=nou,a0=a0,topwid=topwid,logunit=logunit,verbose=verbose, $
        error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  error=0 & rname='p3d_misc_read_params: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) or ~n_params() then begin
    doc_library,'p3d_misc_read_params'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking input & output variables:

  if keyword_set(a0) and ~n_elements(parname) then return

  if ~n_elements(parname) or ~n_elements(name) or $
     n_elements(parvalue) ne n_elements(parname) then begin
    errmsg='PARNAME, PARVALUE, and NAME must all be set; PARVALUE and NAME' + $
           ' must have the same number of elements.'
    goto,error_handler
  endif

  uparfile=0L
  nu=n_elements(uparname)
  if nu gt 0L and nu eq n_elements(uparvalue) then uparfile=1L

  checktype=0L & s=size(type_)
  if s[s[0L]+2L] eq 1L then begin
    if s[s[0L]+1L] ne 7L then begin
      errmsg='TYPE must be set to a scalar string; TYPE=''integer''||' + $
             '''float''.'
      goto,error_handler
    endif
    type=strlowcase(type_)
    if type ne 'integer' and type ne 'float' then begin
      errmsg='TYPE must be set to a scalar string; TYPE=''integer'',' + $
             '''float'', TYPE='+type+' is not valid.'
      goto,error_handler
    endif
    checktype=1L
  endif ;; s[s[0L]+2L] eq 1L

  ;;========================================------------------------------
  ;; Locating NAME within PARNAME:

  parsrc=keyword_set(upo)?' [userparfile].':' [default].'
  idx=strpos(strlowcase(strtrim(parname,2L)),strlowcase(strtrim(name,2L)))
  ok=where(idx eq 0L,count)
  if count eq 1L then value=parvalue[ok[0L]]
  found=~count?0L:1L
  if keyword_set(must_exist) and ~count then begin
    errmsg='The parameter with the name NAME="'+strtrim(name,2L)+'" could ' + $
        'not be found.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Also checking in UPARNAME and UPARVALUE:

  if uparfile then begin
    ok=where(strupcase(strtrim(uparname,2L)) eq $
             strupcase(strtrim(name    ,2L)),count)
    if count ne 0L then begin
      value=uparvalue[ok[0L]]
      parsrc=' [userparfile].'
    endif
  endif ;; uparfile

  ;;========================================------------------------------
  ;; Checking the variable type:

  if found then ovalue=value
  if found and checktype then begin
    errmsg='The read parameter value ('+value+') cannot be converted to a'
    vartype=size(value,/type)
    ok=0L
    case type of
      'integer': begin
        on_ioerror,errorinteger
        value=long(value)
        ok=1L
errorinteger:
        on_ioerror,NULL
        if ~ok then begin
          errmsg+='n integer.'
          goto,error_handler
        endif
      end
      else: begin
        on_ioerror,errorfloat
        value=double(value)
        ok=1L
errorfloat:
        on_ioerror,NULL
        if ~ok then begin
          errmsg+=' decimal value.'
          goto,error_handler
        endif
      end
   endcase ;; type
  endif ;; checktype

  ;;========================================------------------------------
  ;; Logging:

  if found then begin
    tmpstr=strtrim(ovalue,2L) & slen=strlen(tmpstr)
    tmpstr=slen lt 9L?string(format='(a10)',ovalue):tmpstr
    tname=strlen(name) lt 20L?string(name,format='(a20)'):name
    tmp=checktype?'(->'+type+(type eq 'float'?'  ':'')+')':''

    parsrc=keyword_set(nou)?'.':parsrc
    msg='Read '+tname+'="'+tmpstr+'" '+tmp+parsrc
    error=p3d_misc_logger(msg,logunit,loglevel=2,rname=rname,topwid=topwid, $
        verbose=verbose ge 2)
  endif ;; found

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_misc_read_params
