;+
; NAME:
;         p3d_tracing_correctpos
;
;         $Id: p3d_tracing_correctpos.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         This routine refines already calculated spectrum positions (in the
;         cross-dispersion direction). The refining is made by weighting each
;         spectrum position with the cross-dispersion axis profile. Optionally
;         a Gaussian curve, of specified proportions can be used for further
;         weighting (using the GAP argument). Such a Gaussian is always used
;         with the first and last spectrum.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: tracing of spectra on the CCD
;
; CALLING SEQUENCE:
;         p3d_tracing_correctpos,array,oldpos,gap,dist,fwhm,width,niterat, $
;             newpos,integral,topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         array           - A two-dimensional array of floating point type.
;         oldpos          - A one-dimensional array which contains the spectrum
;                           positions which are to be corrected.
;         gap             - If used: a one-dimensional array of floating point
;                           values.
;                           If not used: a scalar integer == -1.
;         dist            - A scalar decimal value which defines the expected
;                           separation of spectra in the cross-dispersion
;                           direction of ARRAY; DIST>0.0..
;         fwhm            - A scalar decimal value, that defines the full
;                           width at half maximum of the Gaussian profile,
;                           which is used when weighting the cross-dispersion
;                           spectrum profile; FWHM>0.0.
;         width           - A scalar decimal value, which specifies the cross-
;                           dispersion profile width; WIDTH>0.0.
;         niterat         - A scalar integer specifying how many times the
;                           weighted spectrum position value is calculated;
;                           NITERAT>=1.
;
; KEYWORD PARAMETERS:
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this  
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         newpos          - A one-dimensional array which contains the
;                           corrected positions of all spectra provided in
;                           OLDPOS.
;         integral        - TBA.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         06.10.2008 - Converted from the original routine correctpos of
;                      Thomas Becker. /CS
;-
PRO p3d_tracing_correctpos,array,oldpos,gap,dist_,fwhm,width,niterat, $
        newpos,integral,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  if !version.release lt 6.2 then message,'IDL Version <6.2. Cannot continue.'
  error=0 & rname='p3d_tracing_correctpos: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_tracing_correctpos'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  s=size(array)
  if s[0L] ne 2L or (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='ARRAY [1] must be a two-dimensional array of floating point type.'
    goto,error_handler
  endif

  v=size(oldpos)
  if v[0L] ne 1L or (v[v[0L]+1L] ge 6L and v[v[0L]+1L] le 11L) then begin
    errmsg='OLDPOS [2] must be a one-dimensional array of floating point type.'
    goto,error_handler
  endif

  sb=size(gap)
  if sb[sb[0L]+1L] ge 6L and sb[sb[0L]+1L] le 11L then begin
    errmsg='GAP [3] must be of floating point type.'
    goto,error_handler
  endif

  if ~sb[0L] and sb[sb[0L]+2L] eq 1L then begin
    if gap ne -1L then begin
      errmsg='GAP [3] must be set to an integer scalar == -1 if it is not ' + $
             'used.'
      goto,error_handler
    endif
  endif else if sb[0L] ne 1L then begin
    errmsg='GAP [3] must be set to an array of floating point values if it' + $
           ' is used.'
    goto,error_handler
  endif
  sg=gap[0] ne -1L?n_elements(gap):0L

  s=size(dist_)
  if s[s[0L]+2L] ne 1L or (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='DIST [4] must be a decimal scalar; >0.0.'
    goto,error_handler
  endif
  if dist_ le 0d0 then begin
    errmsg='DIST [4] must be a decimal scalar; >0.0.'
    goto,error_handler
  endif
  dist=dist_

  s=size(fwhm)
  if s[s[0L]+2L] ne 1L or (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='FWHM [5] must be a decimal scalar; >0.0.'
    goto,error_handler
  endif
  if fwhm le 0d0 then begin
    errmsg='FWHM [5] must be a decimal scalar; >0.0.'
    goto,error_handler
  endif

  s=size(width)
  if s[s[0L]+2L] ne 1L or (s[s[0L]+1L] ge 6L and s[s[0L]+1L] le 11L) then begin
    errmsg='WIDTH [6] must be a decimal scalar.'
    goto,error_handler
  endif
  if width le 0d0 then begin
    errmsg='WIDTH [6] must be a decimal scalar; >0.0.'
    goto,error_handler
  endif

  s=size(niterat)
  if s[s[0L]+2L] ne 1L or (s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L) then begin
    errmsg='NITERAT [7] must be an integer scalar; >1.'
    goto,error_handler
  endif
  if niterat lt 1L then begin
    errmsg='NITERAT [7] must be an integer scalar; >1.'
    goto,error_handler
  endif

  s=size(array)

  ;;========================================------------------------------
  ;; Calculating a generic cross-dispersion-Gaussian profile:

  profsize=ceil(1.5d0*fwhm)
    space0=ceil(ceil(dist)+profsize-oldpos[0L])>0L
    space1=ceil(ceil(dist)+profsize+oldpos[v[1L]-1L]-s[2L]+2L)>0L

  tmp=findgen(ceil(10L*(2L*profsize+1L)+20L))
  profil=exp(-((temporary(tmp)-(10L*profsize+15L))/1d1/ $
               fwhm*2.3d0*alog(2d0))^2)

  pos=oldpos+space0

  ;;========================================------------------------------
  ;; Setting values for the first spectrum:

  gappos_low=round(1d1*(pos[0L]-dist))/1d1
  gappix_low=long(gappos_low)
  gaprem_low=gappos_low-gappix_low

  prof_lower=dblarr(2L*profsize+1L)

  for k=0L,2L*profsize do $
     prof_lower[k]=total(profil[10L*(k-gaprem_low)+10L: $
                                10L*(k-gaprem_low)+19L])
  prof_lower/=max(prof_lower) ;; normalizing the spectrum

  ;;========================================------------------------------
  ;; Setting values for the last spectrum:

  gappos_hig=round(1d1*(pos[v[1L]-1L]+dist))/1d1
  gappix_hig=long(gappos_hig)
  gaprem_hig=gappos_hig-gappix_hig

  prof_higer=dblarr(2L*profsize+1L)
  for k=0L,2L*profsize do $
     prof_higer[k]=total(profil[10L*(k-gaprem_hig)+10L: $
                                10L*(k-gaprem_hig)+19L])
  prof_higer/=max(prof_higer) ;; normalizing the spectrum

  ;;========================================------------------------------
  ;; Looping over all spectrum bins of ARRAY in the dispersion dimension:

    corpos=dblarr(s[1L],v[1L])
  integral=dblarr(v[1L])

  for m=0L,s[1L]-1L do begin

    vec=dblarr(s[2L]+space0+space1)
    vec[space0:s[2L]+space0-1L]=transpose(array[m,*])

    ;; the first spectrum:
    prof=prof_lower*(vec[pos[0L]      ]-vec[gappix_low])
    vec[(gappix_low-profsize):(gappix_low+profsize)]+=temporary(prof)

    ;; the last spectrum:
    prof=prof_higer*(vec[pos[v[1L]-1L]]-vec[gappix_hig])
    vec[(gappix_hig-profsize):(gappix_hig+profsize)]+=temporary(prof)

    ;;========================================------------------------------
    ;; Looping over all elements of the GAP array:

    for k=0L,sg-1L do begin

      if ~k then begin
        left =1L
        if sg gt 1L then right=gap[1L] gt gap[0L]?1L:0L else right=1L
      endif else if k eq sg-1L then begin
        right=1L
        left =gap[sg-2L] lt gap[sg-1L]-1L?1L:0L
      endif else begin
        left =gap[k-1L] lt gap[k]?1L:0L
        right=gap[k+1L] gt gap[k]?1L:0L
      endelse

      if left+right gt 0L then begin

        if left and right then begin
          gappos=double(pos[gap[k]-1L]+pos[gap[k]])/2L
        endif else if left then begin
          gappos=round(1d1*(pos[gap[k]-1L]+dist))/1d1
        endif else begin
          gappos=round(1d1*(pos[gap[k]   ]-dist))/1d1
        endelse

        gappix=long(gappos)
        gaprem=gappos-gappix
        gappeak=(max(vec[pos[gap[k]-1L]-1L:pos[gap[k]-1L]+1L])* left + $
                 max(vec[pos[gap[k]   ]-1L:pos[gap[k]   ]+1L])*right)/ $
                (left+right)-vec[round(gappos)]

        prof=dblarr(2L*profsize+1L)
        for L=0L,2L*profsize do $
           prof[L]=total(profil[10L*(L-gaprem)+10L: $
                                10L*(L-gaprem)+19L])
        prof*=gappeak/max(prof)

        vec[(gappix-profsize):(gappix+profsize)]+=temporary(prof)

      endif ;; left+right gt 0L
    endfor ;; k=0L,sg-1L

    vec=vec>0d0

    ;;========================================------------------------------
    ;; Looping over all spectra. Calculating a central position - in the cross-
    ;; dispersion direction - weighted with the spectrum profile. The value is
    ;; iterated NITERAT times:

    newpos=double(pos)

    for k=0L,v[1L]-1L do begin 
      pixpos=newpos[k]+0.5

      niterat_counter=niterat
      while niterat_counter-- gt 0L do begin

        if pixpos ge width and pixpos lt s[2L]-width then begin
          lower=pixpos-width & lowlon=long(lower) & lowrem=     lower-lowlon
          upper=pixpos+width & upplon=long(upper) & upprem=1d0-(upper-upplon)

          sum=total(vec[lowlon:upplon])
          sum-=lowrem*vec[lowlon]+upprem*vec[upplon]

          tmp=findgen(upplon-lowlon+1L)+lowlon
          weight=total(vec[lowlon:upplon]*temporary(tmp))
          weight-=lowrem*vec[lowlon]*lowlon + $
                  upprem*vec[upplon]*upplon

          pixpos=weight/sum+0.5d0

        endif else begin
          ;; No summing can be made for the first and last spectrum in this
          ;; case:

          pixpos=-1L
        endelse

      endwhile ;; niterat_counter-- gt 0L

      if pixpos eq -1L then begin
        newpos[k]=pos[k]
      endif else begin   
        newpos[k]=pixpos-0.5d0
        integral[k]+=sum>0d0
      endelse

     endfor ;; k=0L,v[1L]-1L

    corpos[m,*]=newpos

  endfor ;; m=0L,s[1L]-1L

  ;;========================================------------------------------
  ;; Storing the weighted average values, using the median value - across the
  ;; dispersion direction:

  newpos=reform(median(corpos,dimension=1L)-space0)
  integral/=v[1L]

  ;;========================================------------------------------
  ;; Logging the performed operations:

  tmp=size(reform(array)) ;; temporary solution
  if tmp[0L] eq 2L then begin
    tmp=logunit[1L] ge 2L? $
        ['  Calculated the following '+strtrim(n_elements(newpos),2L)+ $
         ' corrected positions [px]:', $
         p3d_tracing_findspec_logarray(newpos),'']: $
        ['  Calculated '+strtrim(n_elements(newpos),2L)+ $
         ' corrected positions.']

    msg=['Centering spectra :: Step 1e [calculating corrected positions]', $
         '          Spectrum separation [dist_tr]='+strtrim(dist,2L), $
         '        Cross-dispersion FWHM [fwhm_tr]='+strtrim(fwhm,2L), $
         '  Cross-dispersion width [centervar_tr]='+strtrim(width,2L), $
         '  The position was iterated '+strtrim(niterat,2L)+ $
         ' times [niterat_tr; for all spectra and wavelength bins].', $
         tmp]
    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
  endif ;; tmp[0L] eq 2L

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_tracing_correctpos
