;+
; NAME:
;         p3d_misc_fitsparsplit
;
;         $Id: p3d_misc_fitsparsplit.pro 79 2010-03-04 14:24:25Z christersandin $
;
; PURPOSE:
;         Routine description pending.
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: auxiliary routines
;
; CALLING SEQUENCE:
;         p3d_misc_fitsparsplit,parameter,string__,hdr,comment,maxn=, $
;             topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         parameter       - .
;         string          - .
;         hdr             - .
;         comment         - .
;
; KEYWORD PARAMETERS:
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;         _extra          - Extra keywords passed on to p3d_tracing_trace
;
; OUTPUTS:
;         traces          - The outcome, a trace mask.
;         array           - The raw image data.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         06.10.2008 - Converted from original routines p3de_trace_file of
;                      Thomas Becker and Petra Bhm. /CS
;-
PRO p3d_misc_fitsparsplit,parameter,string__,hdr,comment,maxn=maxn, $
        topwid=topwid,logunit=logunit,verbose=verbose, $
        error=error,debug=debug,help=help
  compile_opt IDL2,hidden

  error=0 & rname='p3d_misc_fitsparsplit: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_misc_fitsparsplit'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  if n_elements(parameter) ne 1L or size(parameter,/type) ne 7L then begin
    errmsg='PARAMETER must be a scalar string, of maximum length 7 characters.'
    goto,error_handler
  endif
  if ~strlen(parameter) or strlen(parameter) gt 7L then begin
    errmsg='PARAMETER must be a scalar string, of maximum length 7 characters.'
    goto,error_handler
  endif

  s=size(string__)
  if s[0L] ne 0L or s[s[0L]+1L] ne 7L then begin
    errmsg='STRING must be set; to a scalar string.'
    goto,error_handler
  endif
  string=string__

  if ~n_elements(hdr) or size(hdr,/type) ne 7L then begin
    errmsg='HDR must be set; to a scalar string array.'
    goto,error_handler
  endif

  if n_elements(comment) ne 1L or size(comment,/type) ne 7L then begin
    errmsg='COMMENT must be set; to a scalar string.'
    goto,error_handler
  endif

  if ~n_elements(maxn) then maxn=35L
  s=size(maxn)
  if s[s[0L]+1L] ge 4L and s[s[0L]+1L] le 11L then begin
    errmsg='MAXN must be of (any) integer type.'
    goto,error_handler
  endif

  ;;========================================------------------------------
  ;; Splitting the string 'string' into several FITS header keywords, if
  ;; necessary:

  fxaddpar,hdr,parameter,strmid(string,0L,maxn),comment

  if strlen(string) gt maxn then begin
    string=strmid(string,maxn) & i=1L
    repeat begin
      str=strmid(string,0,maxn)
      fxaddpar,hdr,parameter+strtrim(i++,2L),str,comment+', continued.'
      string=strmid(string,maxn)
    endrep until string eq ''
  endif

  return

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;; procedure: p3d_misc_fitsparsplit
