;+
; NAME:
;         p3d_extract_prepare_extraction
;
;         $Id: p3d_extract_prepare_extraction.pro 181 2010-04-21 08:44:03Z christersandin $
;
; PURPOSE:
;         This routine loads object data, a cosmic ray mask, a trace mask, and
;         a master bias. The data is checked for consistency in terms of array
;         sizes and used binning parameters. In a second step all data is
;         trimmed of prescan and overscan regions, using the DETSEC keyword.
;
;         The data is returned as is (i.e. no transposing).
;
; AUTHOR:
;         Christer Sandin
;         Astrophysikalisches Institut Potsdam (AIP)
;         An der Sternwarte 16
;         D-14482 Potsdam, GERMANY
;
; COPYRIGHT:
;         p3d: a general data-reduction tool for fiber-fed IFSs
;
;         Copyright 2009,2010 Astrophysikalisches Institut Potsdam (AIP)
;
;         This program is free software; you can redistribute it and/or modify
;         it under the terms of the GNU General Public License as published by
;         the Free Software Foundation; either version 3 of the License, or
;         (at your option) any later version.
;
;         This program is distributed in the hope that it will be useful, but
;         WITHOUT ANY WARRANTY; without even the implied warranty of
;         MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU
;         General Public License for more details.
;
;         You should have received a copy of the GNU General Public License
;         along with this program; if not, see <http://www.gnu.org/licenses>.
;
;         Additional permission under GNU GPL version 3 section 7
;
;         If you modify this Program, or any covered work, by linking or
;         combining it with IDL (or a modified version of that library),
;         containing parts covered by the terms of the IDL license, the
;         licensors of this Program grant you additional permission to convey
;         the resulting work.
;
; CATEGORY:
;         p3d :: spectrum extraction
;
; CALLING SEQUENCE:
;         p3d_extract_prepare_extraction,extractfile,tracefile,kwrdlist, $
;             trace,object,objshift,mbias,bsw,crmaskfile=,crmaskdata=, $
;             userparfile=,xbin=,ybin=,masterbias=,trcshiftfile=, $
;             dispshiftfile=,objshiftfile=,no_shift=,parfile=,xsize=,ysize=, $
;             x0=,y0=,daxis=,topwid=,logunit=,verbose=,error=,/debug,/help
;
; INPUTS:
;         extractfile     - The input raw data image that is to be treated.
;         tracefile       - A corresponding input file for tracing.
;         kwrdlist        - A two-dimensional string array holding the
;                           instrument-specific keywords for a defined set of
;                           keywords which are used to determine the ROI on the
;                           CCD. KWRDLIST must have two columns.
;
; KEYWORD PARAMETERS:
;         crmaskfile      - The filename of an, optional, cosmic ray mask.
;         crmaskdata      - Returns the cosmic ray mask data array where
;                           prescan and overscan regions, if any, have been
;                           removed (only present if CRMASKFILE is set).
;         userparfile     - A scalar string specifying the name of an optional
;                           user parameter file, that could contain the keyword
;                           'detsec'. If it does then the value of that keyword
;                           is used to trim the data.
;                           If there are several 'detsec'-lines in the
;                           file then only the first is used.
;         xbin            - The CCD binning number; x-direction.
;         ybin            - The CCD binning number; y-direction.
;         masterbias      - The data of MASTERBIAS will be used as bias data.
;         trcshiftfile    - n/a.
;         dispshiftfile   - n/a.
;         objshiftfile    - n/a.
;         no_shift        - n/a.
;         parfile         - A string specifying the filename of an existing
;                           file containing instrument-specific parameters.
;         xsize           - Upon return this value is set to the x-size of the
;                           extracted data.
;         ysize           - Upon return this value is set to the y-size of the
;                           extracted data.
;         x0              - Upon return this value is set to the x-offset of
;                           the extracted data.
;         y0              - Upon return this value is set to the y-offset of
;                           the extracted data.
;         daxis [1]       - Specifies the dispersion dimension in the input
;                           images (EXTRACTFILE, TRACEFILE, MASTERBIAS).
;         topwid          - If set, then error messages are displayed using
;                           DIALOG_MESSAGE, using this widget id as
;                           DIALOG_PARENT, instead of MESSAGE.
;         logunit         - Messages are saved to the file pointed to by this
;                           logical file unit, if it is defined.
;         verbose         - Show more information on what is being done.
;         error           - Returns an error code if set.
;         debug           - The error handler is not setup if debug is set.
;         help            - Show this routine documentation, and exit.
;
; OUTPUTS:
;         trace           - Returns the trace mask data array where prescan and
;                           overscan regions, if any, have been removed.
;         object          - Returns the object data array where prescan and
;                           overscan regions, if any, have been removed.
;         objshift        - n/a.
;         mbias           - If MASTERBIAS is set then this variable returns the
;                           master bias data array where prescan and overscan
;                           regions, if any, have been removed.
;         bsw             - n/a.
;
; COMMON BLOCKS:
;         none
;
; SIDE EFFECTS:
;         none
;
; RESTRICTIONS:
;         IDL version 6.2 or higher is required.
;
; MODIFICATION HISTORY:
;         08.10.2008 - Converted from the original routine prepare_extraction  
;                      of Thomas Becker and Petra Bhm. /CS
;         31.05.2009 - Modified the routine to use DETSEC, and included logging
;                      information. /CS
;-
PRO p3d_extract_prepare_extraction,extractfile,tracefile,kwrdlist, $
        trace,object,objshift,mbias,bsw,crmaskfile=crmaskfile, $
        crmaskdata=crmask,userparfile=userparfile,xbin=xbin,ybin=ybin, $
        masterbias=masterbias, $
        trcshiftfile=trcshiftfile,dispshiftfile=dispshiftfile, $
        objshiftfile=objshiftfile,no_shift=no_shift,parfile=parfile, $
        xsize=xsize,ysize=ysize,x0=x0,y0=y0,daxis=daxis,topwid=topwid, $
        logunit=logunit,verbose=verbose,error=error,debug=debug,help=help
  compile_opt hidden,IDL2

  error=0 & rname='p3d_extract_prepare_extraction: '
  if ~n_elements(verbose) then verbose=0
  debug=keyword_set(debug)

  if keyword_set(help) then begin
    doc_library,'p3d_extract_prepare_extraction'
    return
  endif

  ;;========================================------------------------------
  ;; Setting up an error handler:

  if ~debug then begin
    catch,error_status
    if error_status ne 0L then begin
      p3d_misc_errors,error_status,rname=rname,topwid=topwid
      catch,/cancel
      error=-1
      return
    endif
  endif ;; ~debug

  ;;========================================------------------------------
  ;; Checking the input arguments:

  s=size(extractfile)
  if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
    errmsg='EXTRACTFILE must be set; to the name of an existing file.'
    goto,error_handler
  endif
  if ~file_test(extractfile,/regular,/read) then begin
    errmsg='The file EXTRACTFILE "'+extractfile+'" does not exist.'
    goto,error_handler
  endif

  s=size(tracefile)
  if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
    errmsg='TRACEFILE must be set; to the name of an existing file.'
    goto,error_handler
  endif
  if ~file_test(tracefile,/regular,/read) then begin
    errmsg='The file TRACEFILE "'+tracefile+'" does not exist.'
    goto,error_handler
  endif

  sk=size(kwrdlist)
  if sk[sk[0L]+2L] ne 1L or sk[sk[0L]+1L] ne 7L then begin
    errmsg='KWRDLIST must be set; to the name of a file which contains a l' + $
           'ist of instrument-specific keywords.'
    goto,error_handler
  endif

  usembiasfile=0L
  s=size(masterbias)
  if s[s[0L]+2L] ne 0L then begin
    if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
      errmsg='MASTERBIAS, if defined, must be a string, which is pointing ' + $
             'at an existing file.'
      goto,error_handler
    endif
    if ~file_test(masterbias,/regular,/read) then begin
      errmsg='The file MASTERBIAS "'+masterbias+'" does not exist.'
      goto,error_handler
    endif
    usembiasfile=1L
  endif ;; s[s[0L]+2L] ne 0L

  dotraceshif=0L
  s=size(trcshiftfile)
  if s[s[0L]+2L] ne 0L then begin
    if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
      errmsg='TRCSHIFTFILE, if defined, must be a string, which is pointin' + $
             'g at an existing file.'
      goto,error_handler
    endif
    if ~file_test(trcshiftfile,/regular,/read) then begin
      errmsg='The file TRCSHIFTFILE "'+trcshiftfile+'" does not exist.'
      goto,error_handler
    endif

    s=size(objshiftfile)
    if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
      errmsg='OBJSHIFTFILE, if defined, must be a string, which is pointin' + $
             'g at an existing file.'
      goto,error_handler
    endif
    if ~file_test(objshiftfile,/regular,/read) then begin
      errmsg='The file OBJSHIFTFILE "'+objshiftfile+'" does not exist.'
      goto,error_handler
    endif

    dotraceshif=1L
  endif ;; s[s[0L]+2L] ne 0L

  dodispshift=0L
  s=size(dispshiftfile)
  if s[s[0L]+2L] ne 0L then begin
    if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
      errmsg='DISPSHIFTFILE, if defined, must be a string, that points at ' + $
             'an existing file.'
      goto,error_handler
    endif
    if ~file_test(dispshiftfile,/regular,/read) then begin
      errmsg='The file DISPSHIFTFILE "'+dispshiftfile+'" does not exist.'
      goto,error_handler
    endif
    dodispshift=1L
  endif ;; s[s[0L]+2L] ne 0L

  usecrmaskfile=0L
  s=size(crmaskfile)
  if s[s[0L]+2L] ne 0L then begin
    if s[s[0L]+2L] ne 1L or s[s[0L]+1L] ne 7L then begin
      errmsg='CRMASKFILE, if defined, must be a string, which is pointing ' + $
             'at an existing file.'
      goto,error_handler
    endif
    if ~file_test(crmaskfile,/regular,/read) then begin
      errmsg='The file CRMASKFILE "'+crmaskfile+'" does not exist.'
      goto,error_handler
    endif
    usecrmaskfile=1L
  endif ;; s[s[0L]+2L] ne 0L

  no_shift=keyword_set(no_shift)

  if ~n_elements(daxis) then daxis=1L
  sd=size(daxis)
  if sd[sd[0L]+2L] ne 1L or $
    (sd[sd[0L]+1L] ge 4L and sd[sd[0L]+1L] le 11L) then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif
  if daxis ne 1L and daxis ne 2L then begin
    errmsg='DAXIS must be a scalar integer; 1||2.'
    goto,error_handler
  endif
  sid=daxis?2L:1L

  ;;========================================------------------------------
  ;; Reading object data;
  ;;   also reading and checking the binning parameters, and array sizes:

  object=readfits(extractfile,hdr,silent=verbose lt 3,/no_unsigned)
  if max(object) eq min(object) then begin
    errmsg='Empty Image! Will not continue.'
    goto,error_handler
  endif

  p3d_misc_binextract,kwrdlist,hdr,xbin=xbin,ybin=ybin,topwid=topwid, $
      logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return

  xs=fxpar(hdr,'NAXIS1') ;; Number of elements along the x axis
  ys=fxpar(hdr,'NAXIS2') ;; Number of elements along the y axis

  msg='Loading and trimming object data ['+ $
    strtrim(xs,2L)+','+strtrim(ys,2L)+'] "'+ $
    p3d_misc_pathify(extractfile,/dpath)+'".'
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

  ;; Reading tracing data:
  trace=readfits(tracefile,trace_hdr,silent=verbose lt 3,/no_unsigned)

  ;; Testing the dispersion-axis flipping state of the data and the trace file:
  tmp1=fxpar(      hdr,'P3DDFLIP')
  tmp2=fxpar(trace_hdr,'P3DDFLIP')
  if tmp1 ne tmp2 then begin
    errmsg='The dispersion axis flipping of the data file and the trace' + $
           ' mask differ {P3DDFLIP}.'
    goto,error_handler
  endif

  p3d_misc_binextract,kwrdlist,trace_hdr,xbin=trc_xbin,ybin=trc_ybin, $ 
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return

  if trc_xbin ne xbin or trc_ybin ne ybin then begin
    errmsg='The object data binning parameters ['+strtrim(xbin,2L)+','+ $
        strtrim(ybin,2L)+'] do not agree with those of the trace mask ['+ $
        strtrim(trc_xbin,2L)+','+strtrim(trc_ybin,2L)+'].'
    goto,error_handler
  endif

  txs=fxpar(trace_hdr,'NAXIS1')
  if daxis eq 1L and txs ne xs then begin
    errmsg='The object data array x-size ['+strtrim(xs,2L)+ $
      '] does not agree with that of the trace mask ['+strtrim(txs,2L)+'].'
    goto,error_handler
  endif

  tys=fxpar(trace_hdr,'NAXIS2')
  if daxis eq 2L and tys ne ys then begin
    errmsg='The object data array y-size ['+strtrim(ys,2L)+ $
      '] does not agree with that of the trace mask ['+strtrim(tys,2L)+'].'
    goto,error_handler
  endif

  msg='Loading and trimming the trace mask ['+ $
    strtrim(txs,2L)+','+strtrim(tys,2L)+'] "'+ $
    p3d_misc_pathify(tracefile,/dpath)+'".'
  error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)

  ;; Reading the master bias data:
  if usembiasfile then begin
    mbias=readfits(masterbias,mbias_hdr,silent=verbose lt 3,/no_unsigned)

    ;; Testing the dispersion-axis flipping state of the data and the master
    ;; bias file:
    tmp1=fxpar(      hdr,'P3DDFLIP')
    tmp2=fxpar(mbias_hdr,'P3DDFLIP')
    if tmp1 ne tmp2 then begin
      errmsg='The dispersion axis flipping of the data file and the master' + $
             ' bias differ {P3DDFLIP}.'
      goto,error_handler
    endif

    p3d_misc_binextract,kwrdlist,mbias_hdr,xbin=mbias_xbin,ybin=mbias_ybin, $
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then return

    if mbias_xbin ne xbin or mbias_ybin ne ybin then begin
      errmsg='The object data binning parameters ['+strtrim(xbin,2L)+','+ $
        strtrim(ybin,2L)+'] do not agree with those of the master bias ['+ $
        strtrim(mbias_xbin,2L)+','+strtrim(mbias_ybin,2L)+'].'
      goto,error_handler
    endif

    if fxpar(mbias_hdr,'NAXIS1') ne xs or $
       fxpar(mbias_hdr,'NAXIS2') ne ys then begin
      errmsg='The object data array size ['+strtrim(xs,2L)+','+ $
        strtrim(ys,2L)+'] does not agree with that of the master bias ['+ $
        strtrim(fxpar(mbias_hdr,'NAXIS1'),2L)+','+ $
        strtrim(fxpar(mbias_hdr,'NAXIS2'),2L)+'].'
      goto,error_handler
    endif

    msg='Loading and trimming master bias ['+ $
      strtrim(fxpar(mbias_hdr,'NAXIS1'),2L)+','+ $
      strtrim(fxpar(mbias_hdr,'NAXIS2'),2L)+'] "'+ $
      p3d_misc_pathify(masterbias,/dpath)+'".'
    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
  endif ;; usembiasfile

  ;; Reading the cosmic ray mask:
  if usecrmaskfile then begin
    crmask=readfits(crmaskfile,crmask_hdr,silent=verbose lt 3,/no_unsigned)

    p3d_misc_binextract,kwrdlist,crmask_hdr,xbin=crmask_xbin,ybin=crmask_ybin,$
        topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
    if error ne 0 then return

;   if crmask_xbin ne xbin or crmask_ybin ne ybin then begin
;     errmsg='The object data binning parameters ['+strtrim(xbin,2L)+','+ $
;       strtrim(ybin,2L)+'] do not agree with those of the cosmic ray mask' + $
;       ' ['+strtrim(crmask_xbin,2L)+','+strtrim(crmask_ybin,2L)+'].'
;     goto,error_handler
;   endif

    if fxpar(crmask_hdr,'NAXIS1') ne xs or $
       fxpar(crmask_hdr,'NAXIS2') ne ys then begin
      errmsg='The object data array size ['+strtrim(xs,2L)+','+ $
        strtrim(ys,2L)+'] does not agree with that of the cosmic ray mask ['+ $
        strtrim(fxpar(crmask_hdr,'NAXIS1'),2L)+','+ $
        strtrim(fxpar(crmask_hdr,'NAXIS2'),2L)+'].'
      goto,error_handler
    endif

    msg='Loading and trimming the cosmic ray mask ['+ $
      strtrim(fxpar(crmask_hdr,'NAXIS1'),2L)+','+ $
      strtrim(fxpar(crmask_hdr,'NAXIS2'),2L)+'] "'+ $
      p3d_misc_pathify(crmaskfile,/dpath)+'".'
    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
  endif ;; usecrmaskfile


  ;;========================================------------------------------
  ;; Retrieving information about the detector region to use with the arrays:

  p3d_misc_detsec,hdr,kwrdlist,parfile,detsec,userparfile=userparfile, $
      topwid=topwid,logunit=logunit,verbose=verbose,error=error,debug=debug
  if error ne 0 then return

  ;;====================----------
  ;; Reading beam switch-related parameters:

  tmp=p3d_misc_get_hdr_kwrd(kwrdlist,'BS_NROWS',topwid=topwid, $
          logunit=logunit,verbose=verbose,error=error,debug=debug)
  if error ne 0 then return
  bsw=fxpar(hdr,tmp)
  defstr='BS_NROWS ['+strtrim(bsw,2L)+']'
  on_ioerror,format_error
  bsw=long(bsw)
  on_ioerror,NULL

  if bsw ne 0L then begin
    tmp=p3d_misc_get_hdr_kwrd(kwrdlist,'BS_DIR',topwid=topwid, $
            logunit=logunit,verbose=verbose,error=error,debug=debug)
    if error ne 0 then return
    bs_dir=fxpar(hdr,tmp)
    defstr='BS_DIR ['+strtrim(bs_dir,2L)+']'
    on_ioerror,format_error
    bs_dir=long(bs_dir)
    on_ioerror,NULL
    if bsw eq 1L then bsw=-bsw

    msg='Read beam switch parameters; BS_NROWS='+strtrim(bs_nrows,2L)+ $
      ', BS_DIR='+strtrim(bs_dir,2L)+'.'
    error=p3d_misc_logger(msg,logunit,rname=rname,verbose=verbose ge 1)
  endif ;; bsw ne 0L

  
  ;;========================================------------------------------
  ;; Trimming the object, trace, and bias data of prescan and overscan regions:

  sdt=size(detsec,/n_dimensions)
  if sdt eq 1L then begin
    ;; Using only 1 block:

    xsize=xbin*(max(detsec[0L:1L])-min(detsec[0L:1L])+1L)
    x0=xbin*min(detsec[0L:1L])
    ysize=ybin*(max(detsec[2L:3L])-min(detsec[2L:3L])+1L)
    y0=ybin*min(detsec[2L:3L])

    object=object[detsec[0L]:detsec[1L],detsec[2L]:detsec[3L]]
    trace=daxis?trace[detsec[0L]:detsec[1L],*]:trace[*,detsec[2L]:detsec[3L]]
    if usembiasfile then $
       mbias= mbias[detsec[0L]:detsec[1L],detsec[2L]:detsec[3L]]
    if usecrmaskfile then $
      crmask=crmask[detsec[0L]:detsec[1L],detsec[2L]:detsec[3L]]
  endif else begin
    ;; Using several blocks (the min and max functions are present in order to
    ;; keep the data as it is, and not flip it again, as this was already done
    ;; when the data files were combined):

    ;; accounting for pre-scan regions in DETSEC:
    nblocks=(size(detsec,/dimensions))[1L]
    xoff=lonarr(nblocks) & yoff=lonarr(nblocks)
    xmin=min(detsec[0L:1L,*])
    ymin=min(detsec[2L:3L,*])
    for j=0L,nblocks-1L do begin
      if min(detsec[0L:1L,j]) ne xmin then xoff[j]=xmin
      if min(detsec[2L:3L,j]) ne ymin then yoff[j]=ymin
    endfor ;; j=0L,nblocks-1L

    ;; Trimming the data of all pre-scan regions:
    tmp=min(detsec[0L:1L,*],dimension=1L) & minx=min(tmp)
    tmp=max(detsec[0L:1L,*],dimension=1L) & maxx=max(tmp,idx) & maxx-=xoff[idx]
    tmp=min(detsec[2L:3L,*],dimension=1L) & miny=min(tmp)
    tmp=max(detsec[2L:3L,*],dimension=1L) & maxy=max(tmp,idx) & maxy-=yoff[idx]
    object=object[minx:maxx,miny:maxy]

    xsize=xbin*(maxx-minx+1L) & x0=xbin*min(detsec[0L:1L,*])
    ysize=ybin*(maxy-miny+1L) & y0=ybin*min(detsec[2L:3L,*])

    trace=daxis?trace[minx:maxx,*]:trace[*,miny:maxy]
    if usembiasfile  then  mbias= mbias[minx:maxx,miny:maxy]
    if usecrmaskfile then crmask=crmask[minx:maxx,miny:maxy]
  endelse ;; sdt eq 1L


  ;;========================================------------------------------
  ;; Shift boxes [[[not used]]]:

  objshift=fltarr(2L)

  if (dotraceshif or dodispshift) and ~no_shift then begin

    if dotraceshif then begin
      refim=readfits(trcshiftfile,silent=verbose lt 3,/no_unsigned)
      objim=readfits(objshiftfile,silent=verbose lt 3,/no_unsigned)

;;      p3d_tool_box_shifts,refim,objim,curbox,trcshift,maxshift=10L, $
;;          topwid=topwid,logunit=logunit,verbose=verbose,error=error
;;      if error ne 0 then return

      objshift[1L]=trcshift[1L]
    endif

    if dodispshift then begin
      if dispshiftfile ne trcshiftfile or dotraceshif then begin
        dispim=readfits(dispshiftfile,silent=verbose lt 3,/no_unsigned)

;;        p3d_tool_box_shifts,dispim,object,curbox,dispshift,maxshift=10L, $
;;            topwid=topwid,logunit=logunit,verbose=verbose,error=error
;;        if error ne 0 then return

        objshift[0L]=dispshift[0L]
      endif else objshift[0L]=trcshift[0L]
    endif

  endif ;; (dotraceshif or dodispshift) and ~no_shift

  return

format_error:
  errmsg='Could not convert '+defstr+' to long.'
  on_ioerror,NULL

error_handler:
  error=p3d_misc_logger(errmsg,logunit,rname=rname,topwid=topwid, $
      verbose=verbose,/error)
  return
END ;;; procedure: p3d_extract_prepare_extraction
