!/*-----------------------------------------------------------------*/
!/*! 
!  \file f2003multiple.f
!  \brief Example of usage of the ARoME library using the Fortran 2003 interface
!         This example computes an array of Rossiter-McLaughlin effect
!         for different positions of the planet
!         It returns the values computed simulating 3 different techniques:
!           - the CCF technique
!           - the iodine cell technique
!           - the weighted mean (like Ohta et al. 2005)
!
!  \author  G. Boue
!           EXOEarths, Centro de Astrofisica, Universidade do Porto.
!
!   Copyright (C) 2012, CAUP
!   email of the author : gwenael.boue@astro.up.pt
!
!   This work has been supported by the European Research Council/European
!   Community under the FP7 through Starting Grant agreement number 239953, as
!   well as from Fundacao para a Ciencia e a Tecnologia (FCT) through program
!   Ciencia 2007 funded by FCT/MCTES (Portugal) and POPH/FSE (EC), and in the
!   form of grants reference PTDC/CTE-AST/098528/2008, SFRH/BPD/71230/2010, and
!   SFRH/BPD/81084/2011.
!
!
!  History:                                                                
!*/
!/*-----------------------------------------------------------------*/

!/*----------------------------------------------------------------*/
!/* License of the file :
!
!    This program is free software: you can redistribute it and/or modify
!    it under the terms of the GNU General Public License as published by
!    the Free Software Foundation, either version 3 of the License, or
!    (at your option) any later version.
!
!    This program is distributed in the hope that it will be useful,
!    but WITHOUT ANY WARRANTY; without even the implied warranty of
!    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
!    GNU General Public License for more details.
!
!    You should have received a copy of the GNU General Public License
!    along with this program.  If not, see <http://www.gnu.org/licenses/>.
!*/
!/*----------------------------------------------------------------*/

       program main
       use module_arome

       implicit none
       real(8), parameter :: pi = acos(-1.0d0)
       integer, parameter :: ncoord = 20
       real(8) :: c1, c2, c3, c4, beta0, Vsini, sigma0, zeta, Rp
       real(8) :: sma, inc, lambda, anom
       integer :: Kmax
       real(8) :: M_init, M_end
       real(8) :: tab_anom(ncoord)
       real(8), allocatable :: tab_x(:)
       real(8) :: tab_y(ncoord), tab_z(ncoord)
       real(8) :: tab_f(ncoord)
       real(8) :: tab_vp(ncoord)
       real(8) :: tab_betapR(ncoord)
       real(8) :: tab_betapT(ncoord)
       real(8) :: tab_v_CCF(ncoord)
       real(8), allocatable :: tab_v_iodine(:)
       real(8) :: tab_v_mean(ncoord)
       type(C_PTR) :: arome
       integer :: nerror
       integer :: k

       nerror = 0
       !/* tests with some allocatable arrays */
       allocate(tab_x(ncoord))
       allocate(tab_v_iodine(ncoord))
   
       !/* planet orbital parameters */
       sma = 4.0d0              !/* stellar radii */
       inc = 86.0d0*pi/180.0d0  !/* radian */
       
       !/* spin-orbit angle */
       lambda = 30.0d0*pi/180.0d0 !/* radian */
       
       !/* Mean anomaly */
       M_init = 70.0d0*pi/180.0d0
       M_end  = 110.0d0*pi/180.0d0
       
       !/* planet's coordinates */
       do k=1,ncoord
         anom = (M_end-M_init)*(k-1.0)/(ncoord-1.0d0)+M_init
         tab_anom(k) = anom
         tab_x(k) = sma*(-cos(lambda)*cos(anom)+sin(lambda)              &
     &                   *sin(anom)*cos(inc))
         tab_y(k) = sma*(-sin(lambda)*cos(anom)-cos(lambda)              &
     &                   *sin(anom)*cos(inc))
         tab_z(k) = sma*sin(anom)*sin(inc)
       end do
       
       !/* limb-darkening */
       c1 = 0.701d0
       c2 = 0.149d0
       c3 = 0.277d0
       c4 =-0.297d0
       
       !/* line profile */
       beta0  =  2.0d0  !/* width of the non-rotating star */
       Vsini  = 15.0d0  !/* Vsini */
       sigma0 =  8.0d0  !/* width of the best Gaussian fit */
       zeta   =  2.0d0  !/* macro-turbulence parameter */
       Rp     =  0.1d0  !/* radius of the planet */
       Kmax   =    4    !/* order of expansion for the Iodine cell technique */
       
       !/* you can choose to never exit on error */
       call arome_set_continue_on_error()
       
       !/* allocation of the ARoME structure */
       arome = arome_alloc_nl(c1, c2, c3, c4)
       
       !/* or with a quadratic limb-darkening law */
       !//arome = arome_alloc_quad(u1, u2)
       
       !/* set the lineprofile */
       nerror = arome_set_lineprofile(beta0, Vsini, sigma0,             &
     &                                zeta, Kmax, arome)
       
       !/* print error message and exit if status != 0 */
       if (nerror .ne. 0) stop
       
       !/* you can choose to always exit on error */
       call arome_set_exit_on_error()
       
       !/* set the planet parameters */
       nerror = arome_set_planet(Rp, arome)
       
       !/* initialize the techniques you are going to use in the following */
       !/* here we initialize both the CCF and the iodine cell techniques */
       !/* should be done once the planet and line profile parameters are set */
       nerror = arome_init_CCF(arome)
       nerror = arome_init_iodine(arome)
       
       !/* to use the vectorial routines, it is necessary to allocate the necessary space */
       nerror = arome_malloc(ncoord, arome)
       
       !/* compute the flux, vp and betap at given positions (x, y, z) of the planet */
       nerror = arome_mcalc_fvpbetap(tab_x, tab_y, tab_z,               &
     &                               ncoord, arome)
       
       !/* you can have access to the computed flux, subplanet vp, betapR, and betapT */
       nerror = arome_mget_flux(arome, ncoord, tab_f)
       nerror = arome_mget_vp(arome, ncoord, tab_vp)
       nerror = arome_mget_betapR(arome, ncoord, tab_betapR)
       nerror = arome_mget_betapT(arome, ncoord, tab_betapT)
       
       !/* you can change the values of the flux, vp, betapR, betapT, e.g., */
       !/* nerror = arome_mset_flux(own_tab_f, ncoord, arome) */
       
       !/* get the RM signals */
       nerror = arome_mget_RM_CCF(arome, ncoord, tab_v_CCF)
       nerror = arome_mget_RM_iodine(arome, ncoord, tab_v_iodine)
       nerror = arome_mget_RM_mean(arome, ncoord, tab_v_mean)
       
       !/* print the results */
       print '(A9,A9,A9,A9,A9,A9,A9,A9,A9,A9,A9)', "phase",             &
     &        "x", "y", "z", "f", "vp", "betapR", "betapT",             &
     &        "v_CCF", "v_iodine", "v_mean"
       do k=1,ncoord
         print '(SS,F9.6,SP,F9.4,F9.4,F9.4,SS,F9.6,SP,                  &
     &           F9.4,SS,F9.4,F9.4,SP,F9.5,F9.5,F9.5)',                 &
     &           tab_anom(k)/(2.0d0*pi), tab_x(k), tab_y(k), tab_z(k),  &
     &           tab_f(k), tab_vp(k), tab_betapR(k), tab_betapT(k),     &
     &           tab_v_CCF(k), tab_v_iodine(k), tab_v_mean(k)
       end do
       
       !/* release memory */
       deallocate(tab_x)
       deallocate(tab_v_iodine)
       call arome_mfree(arome)
       call arome_free(arome)
       
       end program
