/*-----------------------------------------------------------------*/
/*! 
  \file arome.h 
  \brief public API for ARoME library
        Computes and return analytical Rossiter-McLaughlin signals

  \author  G. Boue
           EXOEarths, Centro de Astrofisica, Universidade do Porto.

   Copyright (C) 2012, CAUP
   email of the author : gwenael.boue@astro.up.pt
   
   This work has been supported by the European Research Council/European
   Community under the FP7 through Starting Grant agreement number 239953, as
   well as from Fundacao para a Ciencia e a Tecnologia (FCT) through program
   Ciencia 2007 funded by FCT/MCTES (Portugal) and POPH/FSE (EC), and in the
   form of grants reference PTDC/CTE-AST/098528/2008, SFRH/BPD/71230/2010, and
   SFRH/BPD/81084/2011.

*/
/*-----------------------------------------------------------------*/

/*----------------------------------------------------------------*/
/* License of the file :

    This program is free software: you can redistribute it and/or modify
    it under the terms of the GNU General Public License as published by
    the Free Software Foundation, either version 3 of the License, or
    (at your option) any later version.

    This program is distributed in the hope that it will be useful,
    but WITHOUT ANY WARRANTY; without even the implied warranty of
    MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
    GNU General Public License for more details.

    You should have received a copy of the GNU General Public License
    along with this program.  If not, see <http://www.gnu.org/licenses/>.
*/
/*----------------------------------------------------------------*/
#ifndef __AROME_H__
#define __AROME_H__

/*----------------------------------------------------------------------------------------------*/
/* definition of the ARoME library version */
/*----------------------------------------------------------------------------------------------*/
/*! version : major number of ARoME library */
#define AROME_VERSION_MAJOR 1
/*! version : minor number of ARoME library */
#define AROME_VERSION_MINOR 0 
/*! version : patch number of ARoME library */
#define AROME_VERSION_PATCH 0


#ifdef __cplusplus
extern "C" {
#endif /* __cplusplus */
   

/*! \def PRAGMA_IVDEP
   Define pragma(ivdep) in order to optimize for-loops
*/
#if defined(HAVE_PRAGMA_IVDEP) && !defined(__cplusplus) 
#define PRAGMA_IVDEP _Pragma("ivdep")
#else
#define PRAGMA_IVDEP 
#endif /*HAVE_PRAGMA_IVDEP*/


/********************************/
/* constant definitions         */
/********************************/
#ifndef fatalerror
#define fatalerror(...) {fprintf(stderr, __VA_ARGS__); exit(EXIT_FAILURE);}
#endif /* fatalerror */

#define _arome_iQuad     0
#define _arome_iNL       1

#define _arome_iCCF      1
#define _arome_iIodine   2

#define _arome_iflux     1
#define _arome_ivp       2
#define _arome_ibetapR   4
#define _arome_ibetapT   8

#define _arome_iNULL     0
#define _arome_iExit     1
#define _arome_iContinue 2
#define _arome_iUserfunc 3
   
#define _arome_iEO       0
#define _arome_iKH       1

#ifndef M_PI
#define M_PI     3.14159265358979323846  /* pi */
#endif /* M_PI */

#ifndef M_PI_2
# define M_PI_2		1.57079632679489661923	/* pi/2 */
#endif /* M_PI_2 */
   
/********************************/
/* Orbit strucutre       */
/********************************/
typedef struct sorbit t_orbit;

/*!
  Orbit parameters
*/
struct sorbit
{
   double per         ; /*!< orbital period in days */
   double sma         ; /*!< semi-major axis in stellar radius */
   double ecc         ; /*!< eccentricity */  
   double om          ; /*!< argument of periastron in rad */
   double k           ; /*!< e*cos(om) */
   double h           ; /*!< e*sin(om) */
   double inc         ; /*!< inclination in rad */
   double lambda      ; /*!< projected spin-orbit angle in rad */
   int    type        ; /*!< iEO (e, om) or iKH (k,h) */
   int    isset       ; /*!< =1 if the orbit parameters are set */
};

/********************************/
/* ARoME structure       */
/********************************/
typedef struct sarome t_arome;

/*!
  ARoME parameters
*/
struct sarome
{
  /* planet */
  double Rp            ; /*!< radius of the planet in stellar radius */
  
  /* line profile */
  double Vsini         ; /*!< rotation velocity of the star in km/s */
  double beta0         ; /*!< width of the non-rotating star in km/s */
  double sigma0        ; /*!< width of the rotating star in km/s */
  double zeta          ; /*!< macroturbulence parameter in km/s */
  int    rotational_Kmax  ; /*!< max order for iodine technique */
  
  /* internal parameters */
  int    type_technique; /*!< iCCF or iIodine */
  double Gaussfit_a0   ; /*!< amplitude a0 of the Gaussian fit */
  double iodine_den    ; /*!< denominator of the Iodine RM signal */
  double flux          ; /*!< flux ratio covered by the planet */
  double vp            ; /*!< subplanet velocity */
  double betapR        ; /*!< radial dispersion of the subplanet profile */
  double betapT        ; /*!< tangential dispersion of the subplanet profile */
  double *mflux        ; /*!< array of flux */
  double *mvp          ; /*!< array of vp */
  double *mbetapR      ; /*!< array of betapR */
  double *mbetapT      ; /*!< array of betapT */
  int    mdim          ; /*!< size of the arrays */
  
  /* flags */
  int    planet_flag        ; /*!< =1 if planet parameters are set */
  int    lineprofile_flag   ; /*!< =1 if line profile parameters are set */
  int    fvpbetap_flag      ; /*!< =0--7 depending on the number of set parameters */
  int    RM_init_flag       ; /*!< =0,1,2,3 check if RM denominators are computed */
  int    mfvpbetap_flag     ; /*!< =0--7 depending on the number of set parameters */
  int    malloc_flag        ; /*!< =1 if mflux, mvp, mbetapR and mbetapT are allocated */
  
  /* limb-darkening */
  int    type_limb     ; /*!< type of limb-darkening (iQuad, iNL) */
  int    nlimb         ; /*!< number of limb coef */
  double *limb_coef    ; /*!< limb-darkening law is  */
  int    *limb_pow     ; /*!< I = sum limb_coef[n] * mu**(limb_pow[n]/2) */
  double *kern_coef    ; /*!< rotational kernel coefficients  */
  
  /* numerical recipes parameters */
  int    gcf_ITMAX     ; /*!< maximal iteration in function gcf (should be 100) */
  double gcf_EPS       ; /*!< exit condition in function gcf (should be 3.0e-7) */
  double gcf_FPMIN     ; /*!< minimum floating point value in function gcf (should be 1.0e-30) */
  int    gser_ITMAX    ; /*!< maximal iteration in function gser (should be 100) */
  double gser_EPS      ; /*!< exit condition in function gser (should be 3.0e-7) */
  double qtrap_EPS     ; /*!< exit condition in function qtrap (should be 1.0e-5) */
  int    qtrap_JMAX    ; /*!< maximal iteration in function qtrap (should be 20) */
};

/********************************/
/* global functions             */
/********************************/
t_arome * arome_alloc_quad(double u1, double u2);
t_arome * arome_alloc_nl(double c1, double c2, double c3, double c4);
int      arome_reset_quad(double u1, double u2, t_arome * const parome);
int      arome_reset_nl(double c1, double c2, double c3, double c4, t_arome * const parome);
void     arome_free(t_arome * parome);
int      arome_set_lineprofile(double beta0, double Vsini, double sigma0, 
                               double zeta, int Kmax, t_arome * const parome);
int      arome_set_planet(double Rp, t_arome * const parome);
int      arome_init_CCF(t_arome * const parome);
int      arome_init_iodine(t_arome * const parome);
int      arome_calc_fvpbetap(double x, double y, double z, t_arome * const parome);
double   arome_get_RM_CCF(t_arome * const parome);
double   arome_get_RM_iodine(t_arome * const parome);
double   arome_get_RM_mean(t_arome * const parome);
int      arome_get_RM_CCF_e(t_arome * const parome, double *res);
int      arome_get_RM_iodine_e(t_arome * const parome, double *res);
int      arome_get_RM_mean_e(t_arome * const parome, double *res);
double   arome_get_flux(t_arome * const parome);
double   arome_get_vp(t_arome * const parome);
double   arome_get_betapR(t_arome * const parome);
double   arome_get_betapT(t_arome * const parome);
int      arome_set_flux(double f, t_arome * const parome);
int      arome_set_vp(double vp, t_arome * const parome);
int      arome_set_betapR(double betap, t_arome * const parome);
int      arome_set_betapT(double betap, t_arome * const parome);
void     arome_set_exit_on_error(void);
void     arome_set_continue_on_error(void);
void     arome_set_func_on_error(void (*userfunc)(const char *));
int      arome_set_gcf_ITMAX(int ITMAX, t_arome * const parome);
int      arome_set_gcf_EPS(double EPS, t_arome * const parome);
int      arome_set_gcf_FPMIN(double FPMIN, t_arome * const parome);
int      arome_set_gser_ITMAX(int ITMAX, t_arome * const parome);
int      arome_set_gser_EPS(double EPS, t_arome * const parome);
int      arome_set_qtrap_EPS(double EPS, t_arome * const parome);
int      arome_set_qtrap_JMAX(int JMAX, t_arome * const parome);
int      arome_malloc(int n, t_arome * const parome);
void     arome_mfree(t_arome * const parome);
int      arome_mcalc_fvpbetap(double *x, double *y, double *z, int n, t_arome * const parome);
int      arome_mget_flux(t_arome * const parome, int n, double *flux);
int      arome_mget_vp(t_arome * const parome, int n, double *vp);
int      arome_mget_betapR(t_arome * const parome, int n, double *betapR);
int      arome_mget_betapT(t_arome * const parome, int n, double *betapT);
int      arome_mset_flux(double *f, int n, t_arome * const parome);
int      arome_mset_vp(double *vp, int n, t_arome * const parome);
int      arome_mset_betapR(double *betapR, int n, t_arome * const parome);
int      arome_mset_betapT(double *betapT, int n, t_arome * const parome);
int      arome_mget_RM_CCF(t_arome * const parome, int n, double *res);
int      arome_mget_RM_iodine(t_arome * const parome, int n, double *res);
int      arome_mget_RM_mean(t_arome * const parome, int n, double *res);

/************************************/
/* function for orbit computations  */
/************************************/
t_orbit * arome_new_orbit();
int       arome_set_orbit_eo(double per, double sma, double ecc, double om, 
                             double inc, double lambda, t_orbit * const orbit);
int       arome_set_orbit_kh(double per, double sma, double k, double h, 
                             double inc, double lambda, t_orbit * const orbit);
double    arome_get_orbit_transit_time(t_orbit * const orbit);
int       arome_get_orbit_xyz(t_orbit * const orbit, double t, double *x, double *y, double *z);
int       arome_mget_orbit_xyz(t_orbit * const orbit, double *t, int n, double *x, double *y, double *z);
void      arome_free_orbit(t_orbit * const orbit);

#ifdef __cplusplus
}
#endif /* __cplusplus */

#endif /* __AROME_H__ */
